<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\User;
use App\Models\Company;
use App\Models\Job;
use App\Models\Application;
use App\Models\Skill;
use App\Models\UserSkill;
use App\Models\Assessment;
use App\Models\AssessmentResult;
use App\Models\Subscription;
use App\Models\PaymentTransaction;
use App\Models\Resume;
use App\Models\UserProfile;

class DatabaseIntegrityChecker extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'db:check-integrity {--fix : Automatically fix issues where possible}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check database integrity and identify potential SQL errors';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Starting Database Integrity Check...');

        $issues = [];
        $autoFixable = [];

        // Check table existence
        $this->checkTableExistence($issues);

        // Check foreign key constraints
        $this->checkForeignKeyConstraints($issues, $autoFixable);

        // Check data consistency
        $this->checkDataConsistency($issues, $autoFixable);

        // Check column types and constraints
        $this->checkColumnConstraints($issues);

        // Check for orphaned records
        $this->checkOrphanedRecords($issues, $autoFixable);

        // Check for duplicate data
        $this->checkDuplicateData($issues);

        // Check for invalid data formats
        $this->checkDataFormats($issues, $autoFixable);

        // Report results
        $this->reportResults($issues, $autoFixable);

        if ($this->option('fix') && !empty($autoFixable)) {
            $this->fixAutoFixableIssues($autoFixable);
        }

        return count($issues) > 0 ? 1 : 0;
    }

    private function checkTableExistence(&$issues)
    {
        $requiredTables = [
            'users', 'companies', 'jobs', 'applications', 'resumes',
            'subscriptions', 'skills', 'user_skills', 'education_levels',
            'assessments', 'assessment_results', 'payment_transactions',
            'roles', 'permissions', 'role_user', 'permission_role'
        ];

        $optionalTables = [
            'user_profiles', 'company_domains', 'saved_searches'
        ];

        foreach ($requiredTables as $table) {
            if (!Schema::hasTable($table)) {
                $issues[] = "Missing required table: {$table}";
            }
        }

        foreach ($optionalTables as $table) {
            if (!Schema::hasTable($table)) {
                $issues[] = "Missing optional table: {$table} (may be created by future migrations)";
            }
        }
    }

    private function checkForeignKeyConstraints(&$issues, &$autoFixable)
    {
        // Check companies.owner_user_id references users.id
        $orphanedCompanies = DB::table('companies')
            ->leftJoin('users', 'companies.owner_user_id', '=', 'users.id')
            ->whereNotNull('companies.owner_user_id')
            ->whereNull('users.id')
            ->count();

        if ($orphanedCompanies > 0) {
            $issues[] = "Found {$orphanedCompanies} companies with invalid owner_user_id references";
            $autoFixable[] = [
                'type' => 'foreign_key',
                'table' => 'companies',
                'column' => 'owner_user_id',
                'action' => 'set_null'
            ];
        }

        // Check jobs.company_id references companies.id
        $orphanedJobs = DB::table('jobs')
            ->leftJoin('companies', 'jobs.company_id', '=', 'companies.id')
            ->whereNull('companies.id')
            ->count();

        if ($orphanedJobs > 0) {
            $issues[] = "Found {$orphanedJobs} jobs with invalid company_id references";
        }

        // Check applications foreign keys
        $orphanedApplications = DB::table('applications')
            ->leftJoin('jobs', 'applications.job_id', '=', 'jobs.id')
            ->leftJoin('users', 'applications.user_id', '=', 'users.id')
            ->whereNull('jobs.id')
            ->orWhereNull('users.id')
            ->count();

        if ($orphanedApplications > 0) {
            $issues[] = "Found {$orphanedApplications} applications with invalid job_id or user_id references";
        }
    }

    private function checkDataConsistency(&$issues, &$autoFixable)
    {
        // Check job salary ranges
        $invalidSalaries = DB::table('jobs')
            ->whereNotNull('salary_min')
            ->whereNotNull('salary_max')
            ->whereColumn('salary_min', '>', 'salary_max')
            ->count();

        if ($invalidSalaries > 0) {
            $issues[] = "Found {$invalidSalaries} jobs where salary_min > salary_max";
            $autoFixable[] = [
                'type' => 'salary_range',
                'count' => $invalidSalaries,
                'action' => 'swap_values'
            ];
        }

        // Check application deadlines (only if the column exists)
        if (Schema::hasColumn('jobs', 'application_deadline')) {
            $expiredJobs = DB::table('jobs')
                ->where('status', 'active')
                ->whereNotNull('application_deadline')
                ->where('application_deadline', '<', now())
                ->count();

            if ($expiredJobs > 0) {
                $issues[] = "Found {$expiredJobs} active jobs with expired application deadlines";
                $autoFixable[] = [
                    'type' => 'expired_deadlines',
                    'count' => $expiredJobs,
                    'action' => 'update_status'
                ];
            }
        }
    }

    private function checkColumnConstraints(&$issues)
    {
        // Check enum values
        $invalidStatuses = DB::table('jobs')
            ->whereNotIn('status', ['active', 'inactive', 'pending', 'expired'])
            ->count();

        if ($invalidStatuses > 0) {
            $issues[] = "Found {$invalidStatuses} jobs with invalid status values";
        }

        $invalidAppStatuses = DB::table('applications')
            ->whereNotIn('status', ['pending', 'reviewing', 'shortlisted', 'interviewed', 'offered', 'rejected', 'withdrawn'])
            ->count();

        if ($invalidAppStatuses > 0) {
            $issues[] = "Found {$invalidAppStatuses} applications with invalid status values";
        }
    }

    private function checkOrphanedRecords(&$issues, &$autoFixable)
    {
        // Check for user profiles without users (only if table exists)
        if (Schema::hasTable('user_profiles')) {
            $orphanedProfiles = DB::table('user_profiles')
                ->leftJoin('users', 'user_profiles.user_id', '=', 'users.id')
                ->whereNull('users.id')
                ->count();

            if ($orphanedProfiles > 0) {
                $issues[] = "Found {$orphanedProfiles} orphaned user profiles";
                $autoFixable[] = [
                    'type' => 'orphaned_records',
                    'table' => 'user_profiles',
                    'count' => $orphanedProfiles,
                    'action' => 'delete'
                ];
            }
        }

        // Check for resumes without users
        $orphanedResumes = DB::table('resumes')
            ->leftJoin('users', 'resumes.user_id', '=', 'users.id')
            ->whereNull('users.id')
            ->count();

        if ($orphanedResumes > 0) {
            $issues[] = "Found {$orphanedResumes} orphaned resumes";
            $autoFixable[] = [
                'type' => 'orphaned_records',
                'table' => 'resumes',
                'count' => $orphanedResumes,
                'action' => 'delete'
            ];
        }
    }

    private function checkDuplicateData(&$issues)
    {
        // Check for duplicate user skills
        $duplicateSkills = DB::table('user_skills')
            ->select('user_id', 'skill_id', DB::raw('COUNT(*) as count'))
            ->groupBy('user_id', 'skill_id')
            ->having('count', '>', 1)
            ->get();

        if ($duplicateSkills->count() > 0) {
            $issues[] = "Found " . $duplicateSkills->count() . " duplicate user-skill associations";
        }

        // Check for duplicate company domains (only if table exists)
        if (Schema::hasTable('company_domains')) {
            $duplicateDomains = DB::table('company_domains')
                ->select('domain', DB::raw('COUNT(*) as count'))
                ->groupBy('domain')
                ->having('count', '>', 1)
                ->get();

            if ($duplicateDomains->count() > 0) {
                $issues[] = "Found " . $duplicateDomains->count() . " duplicate company domains";
            }
        }
    }

    private function checkDataFormats(&$issues, &$autoFixable)
    {
        // Check email formats
        $invalidEmails = DB::table('users')
            ->whereNotNull('email')
            ->where('email', 'NOT REGEXP', '^[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}$')
            ->count();

        if ($invalidEmails > 0) {
            $issues[] = "Found {$invalidEmails} users with invalid email formats";
        }

        // Check phone number formats (basic check) - only if table exists
        if (Schema::hasTable('user_profiles')) {
            $invalidPhones = DB::table('user_profiles')
                ->whereNotNull('phone')
                ->where('phone', 'NOT REGEXP', '^[0-9+\-\s()]+$')
                ->count();

            if ($invalidPhones > 0) {
                $issues[] = "Found {$invalidPhones} user profiles with invalid phone formats";
            }
        }
    }

    private function reportResults($issues, $autoFixable)
    {
        if (empty($issues)) {
            $this->info('✅ No database integrity issues found!');
            return;
        }

        $this->error("❌ Found " . count($issues) . " database integrity issues:");

        foreach ($issues as $issue) {
            $this->line("  • {$issue}");
        }

        if (!empty($autoFixable)) {
            $this->warn("\n🔧 " . count($autoFixable) . " issues can be automatically fixed:");
            foreach ($autoFixable as $fix) {
                $this->line("  • " . ucfirst($fix['type']) . " issue in " . ($fix['table'] ?? 'unknown'));
            }

            if (!$this->option('fix')) {
                $this->line("\nRun with --fix flag to automatically resolve these issues.");
            }
        }
    }

    private function fixAutoFixableIssues($autoFixable)
    {
        $this->info('🔧 Fixing auto-fixable issues...');

        foreach ($autoFixable as $fix) {
            switch ($fix['type']) {
                case 'foreign_key':
                    if ($fix['action'] === 'set_null') {
                        DB::table($fix['table'])
                            ->leftJoin('users', "{$fix['table']}.{$fix['column']}", '=', 'users.id')
                            ->whereNotNull("{$fix['table']}.{$fix['column']}")
                            ->whereNull('users.id')
                            ->update(["{$fix['table']}.{$fix['column']}" => null]);
                        $this->line("  ✅ Fixed foreign key issues in {$fix['table']}");
                    }
                    break;

                case 'salary_range':
                    // Swap min and max salaries where min > max
                    $jobs = DB::table('jobs')
                        ->whereNotNull('salary_min')
                        ->whereNotNull('salary_max')
                        ->whereColumn('salary_min', '>', 'salary_max')
                        ->get();

                    foreach ($jobs as $job) {
                        DB::table('jobs')
                            ->where('id', $job->id)
                            ->update([
                                'salary_min' => $job->salary_max,
                                'salary_max' => $job->salary_min
                            ]);
                    }
                    $this->line("  ✅ Fixed {$fix['count']} salary range issues");
                    break;

                case 'expired_deadlines':
                    if (Schema::hasColumn('jobs', 'application_deadline')) {
                        DB::table('jobs')
                            ->where('status', 'active')
                            ->whereNotNull('application_deadline')
                            ->where('application_deadline', '<', now())
                            ->update(['status' => 'expired']);
                        $this->line("  ✅ Updated status for {$fix['count']} expired jobs");
                    } else {
                        $this->line("  ⚠️ Skipped expired_deadlines fix because jobs.application_deadline column does not exist");
                    }
                    break;

                case 'orphaned_records':
                    if ($fix['action'] === 'delete') {
                        DB::table($fix['table'])
                            ->leftJoin('users', "{$fix['table']}.user_id", '=', 'users.id')
                            ->whereNull('users.id')
                            ->delete();
                        $this->line("  ✅ Deleted {$fix['count']} orphaned records from {$fix['table']}");
                    }
                    break;
            }
        }

        $this->info('✅ Auto-fix completed!');
    }
}