<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Services\MesombPaymentService;

class AdPaymentController extends Controller
{
    private MesombPaymentService $mesomb;

    public function __construct(MesombPaymentService $mesomb)
    {
        $this->mesomb = $mesomb;
        $this->middleware(['auth','verified']);
    }

    /**
     * Initiate ad payment: create ad (pending) and ad_transaction then call Mesomb
     */
    public function initiate(Request $request)
    {
        $data = $request->validate([
            'ad_plan_id' => 'required|exists:ad_plans,id',
            'title' => 'nullable|string|max:255',
            'target_url' => 'nullable|url',
            'start_at' => 'nullable|date',
            'phone' => 'required|string',
            'image' => 'nullable|image|max:5120',
        ]);

        $plan = DB::table('ad_plans')->where('id', $data['ad_plan_id'])->first();
        if (!$plan) {
            return response()->json(['error' => 'Plan not found'], 404);
        }

        // Create ad (pending_payment)
        $imagePath = null;
        if ($request->hasFile('image')) {
            try {
                $imagePath = $request->file('image')->store('ads', 'public');
            } catch (\Exception $e) {
                Log::warning('Failed to store ad image', ['error' => $e->getMessage()]);
            }
        }

        $adId = DB::table('ads')->insertGetId([
            'user_id' => $request->user()->id,
            'ad_plan_id' => $plan->id,
            'title' => $data['title'] ?? null,
            'target_url' => $data['target_url'] ?? null,
            'image_path' => $imagePath,
            'start_at' => $data['start_at'] ?? null,
            'status' => 'pending_payment',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // Create ad_transaction record (pending)
        $transactionId = DB::table('ad_transactions')->insertGetId([
            'ad_id' => $adId,
            'user_id' => $request->user()->id,
            'provider' => 'mesomb',
            'amount_cents' => $plan->price_cents,
            'currency' => 'XAF',
            'status' => 'pending',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // Initiate Mesomb payment
        $amount = ($plan->price_cents / 100); // Mesomb expects base amount
        $payload = [
            'amount' => $amount,
            'currency' => 'XAF',
            'phone' => $data['phone'],
            'reference' => 'AD_' . $transactionId . '_' . now()->timestamp,
            'reason' => 'Ad purchase on HireCameroon',
            'email' => $request->user()->email ?? null,
            'name' => $request->user()->name ?? null,
            'callback_url' => config('app.url') . '/webhooks/mesomb'
        ];

        $result = $this->mesomb->initiatePayment($payload);

        if (!$result['success']) {
            Log::error('Mesomb initiation failed', ['result' => $result, 'ad_id' => $adId]);
            // mark transaction failed
            DB::table('ad_transactions')->where('id', $transactionId)->update(['status' => 'failed', 'raw_payload' => json_encode($result), 'updated_at' => now()]);
            return response()->json(['error' => 'Payment initiation failed', 'detail' => $result['error'] ?? null], 500);
        }

        // Update ad_transaction with provider reference/payment_id
        DB::table('ad_transactions')->where('id', $transactionId)->update([
            'provider_reference' => $result['reference'] ?? null,
            'raw_payload' => json_encode($result),
            'updated_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'payment_url' => $result['payment_url'] ?? null,
            'instructions' => $result['instructions'] ?? null,
            'ad_id' => $adId,
            'transaction_id' => $transactionId,
        ]);
    }
}
