<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ad;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class AdController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $ads = Ad::with('creator')
                 ->orderBy('created_at', 'desc')
                 ->paginate(20);

        return view('admin.ads.index', compact('ads'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.ads.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'link_url' => 'nullable|url',
            'type' => ['required', Rule::in(['banner', 'floating', 'popup', 'sidebar'])],
            'position' => ['nullable', Rule::in(['top', 'bottom', 'left', 'right', 'center'])],
            'status' => ['required', Rule::in(['active', 'inactive', 'scheduled'])],
            'start_date' => 'nullable|date|after_or_equal:today',
            'end_date' => 'nullable|date|after:start_date',
            'budget' => 'nullable|numeric|min:0',
            'targeting' => 'nullable|array',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('ads', 'public');
            $validated['image_url'] = Storage::disk('public')->url($imagePath);
        }

        $validated['created_by'] = auth()->id();

        Ad::create($validated);

        return redirect()->route('admin.ads.index')
                        ->with('success', 'Ad created successfully.');
    }

    /**
     * Display the specified resource.
     *
     * @param  Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function show(Ad $ad)
    {
        $ad->load('creator');

        return view('admin.ads.show', compact('ad'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function edit(Ad $ad)
    {
        return view('admin.ads.edit', compact('ad'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Ad $ad)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'link_url' => 'nullable|url',
            'type' => ['required', Rule::in(['banner', 'floating', 'popup', 'sidebar'])],
            'position' => ['nullable', Rule::in(['top', 'bottom', 'left', 'right', 'center'])],
            'status' => ['required', Rule::in(['active', 'inactive', 'scheduled'])],
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'budget' => 'nullable|numeric|min:0',
            'targeting' => 'nullable|array',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($ad->image_url) {
                $oldPath = str_replace('/storage/', '', $ad->image_url);
                Storage::disk('public')->delete($oldPath);
            }

            $imagePath = $request->file('image')->store('ads', 'public');
            $validated['image_url'] = Storage::disk('public')->url($imagePath);
        }

        $ad->update($validated);

        return redirect()->route('admin.ads.index')
                        ->with('success', 'Ad updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function destroy(Ad $ad)
    {
        // Delete associated image
        if ($ad->image_url) {
            $imagePath = str_replace('/storage/', '', $ad->image_url);
            Storage::disk('public')->delete($imagePath);
        }

        $ad->delete();

        return redirect()->route('admin.ads.index')
                        ->with('success', 'Ad deleted successfully.');
    }

    /**
     * Toggle ad status.
     *
     * @param  Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function toggleStatus(Ad $ad)
    {
        $ad->update([
            'status' => $ad->status === 'active' ? 'inactive' : 'active'
        ]);

        return redirect()->back()
                        ->with('success', 'Ad status updated successfully.');
    }
}
