<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\CommunicationService;
use App\Services\Email\EmailService;
use App\Services\Sms\SmsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class CommunicationController extends Controller
{
    protected CommunicationService $communicationService;
    protected EmailService $emailService;
    protected SmsService $smsService;

    public function __construct(CommunicationService $communicationService, EmailService $emailService, SmsService $smsService)
    {
        $this->communicationService = $communicationService;
        $this->emailService = $emailService;
        $this->smsService = $smsService;
    }

    /**
     * Display communication dashboard
     */
    public function index()
    {
        $stats = $this->communicationService->getStats();
        $recentLogs = DB::table('communication_logs')
            ->orderBy('created_at', 'desc')
            ->limit(50)
            ->get();
            
        $providerStatus = [
            'email' => $this->getEmailProviderStatus(),
            'sms' => $this->getSmsProviderStatus(),
        ];

        return view('admin.communication.index', compact('stats', 'recentLogs', 'providerStatus'));
    }

    /**
     * Display communication settings
     */
    public function settings()
    {
        $settings = DB::table('communication_settings')->get()->keyBy('setting_key');
        $config = config('communication');
        
        return view('admin.communication.settings', compact('settings', 'config'));
    }

    /**
     * Update communication settings
     */
    public function updateSettings(Request $request)
    {
        $validated = $request->validate([
            'communication.default' => 'required|in:email,sms',
            'communication.providers.email.default' => 'required|in:smtp,phpmailer,mailgun,ses,postmark',
            'communication.providers.sms.default' => 'required|in:twilio,messagebird,nexmo,africatalking',
            'communication.throttling.emails_per_hour' => 'integer|min:1|max:1000',
            'communication.throttling.sms_per_hour' => 'integer|min:1|max:100',
            'admin.email_notifications' => 'boolean',
            'admin.sms_notifications' => 'boolean',
            'admin.weekly_reports' => 'boolean',
            'admin.critical_alerts' => 'boolean',
        ]);

        // Update settings in database
        foreach ($validated as $key => $value) {
            DB::table('communication_settings')->updateOrInsert(
                ['setting_key' => $key],
                [
                    'setting_value' => is_bool($value) ? ($value ? '1' : '0') : $value,
                    'type' => is_bool($value) ? 'boolean' : 'string',
                    'updated_at' => now(),
                ]
            );
        }

        // Clear cache to force config reload
        Cache::forget('communication_settings');

        return redirect()->route('admin.communication.settings')
            ->with('success', 'Communication settings updated successfully.');
    }

    /**
     * Test email configuration
     */
    public function testEmail(Request $request)
    {
        $provider = $request->input('provider', 'smtp');
        $result = $this->emailService->testConfiguration($provider);

        if ($request->wantsJson()) {
            return response()->json($result);
        }

        return back()->with($result['success'] ? 'success' : 'error', $result['message']);
    }

    /**
     * Test SMS configuration
     */
    public function testSms(Request $request)
    {
        $provider = $request->input('provider', 'twilio');
        $result = $this->smsService->testConfiguration($provider);

        if ($request->wantsJson()) {
            return response()->json($result);
        }

        return back()->with($result['success'] ? 'success' : 'error', $result['message']);
    }

    /**
     * Send test message
     */
    public function sendTestMessage(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|in:email,sms',
            'provider' => 'required|string',
            'recipient' => 'required|email',
            'phone' => 'nullable|string',
        ]);

        $data = [
            'message' => 'This is a test message from HireCameroon Admin Panel',
            'user' => auth()->user(),
            'timestamp' => now(),
        ];

        if ($validated['type'] === 'email') {
            $success = $this->emailService->{'sendWith' . ucfirst($validated['provider'])}(
                'test', $data, $validated['recipient']
            );
            $channel = 'email';
        } else {
            $success = $this->smsService->{'sendWith' . ucfirst($validated['provider'])}(
                'test', $data, $validated['phone'] ?: $validated['recipient']
            );
            $channel = 'sms';
        }

        $message = $success ? 'Test message sent successfully!' : 'Failed to send test message.';
        $type = $success ? 'success' : 'error';

        if ($request->wantsJson()) {
            return response()->json(['success' => $success, 'message' => $message]);
        }

        return back()->with($type, $message);
    }

    /**
     * Display communication logs
     */
    public function logs(Request $request)
    {
        $query = DB::table('communication_logs')->orderBy('created_at', 'desc');

        if ($request->filled('channel')) {
            $query->where('channel', $request->channel);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('provider')) {
            $query->where('provider', $request->provider);
        }

        if ($request->filled('date_from')) {
            $query->where('sent_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('sent_at', '<=', $request->date_to . ' 23:59:59');
        }

        $logs = $query->paginate(50);

        return view('admin.communication.logs', compact('logs'));
    }

    /**
     * Get communication analytics
     */
    public function analytics(Request $request)
    {
        $period = $request->input('period', '30 days');
        $days = match($period) {
            '7 days' => 7,
            '30 days' => 30,
            '90 days' => 90,
            default => 30,
        };

        $since = now()->subDays($days);

        $analytics = [
            'total_messages' => DB::table('communication_logs')
                ->where('sent_at', '>=', $since)->count(),
            'emails_sent' => DB::table('communication_logs')
                ->where('channel', 'email')->where('sent_at', '>=', $since)->count(),
            'sms_sent' => DB::table('communication_logs')
                ->where('channel', 'sms')->where('sent_at', '>=', $since)->count(),
            'success_rate' => $this->calculateSuccessRate($since),
            'cost_analysis' => $this->calculateCostAnalysis($since),
            'provider_performance' => $this->getProviderPerformance($since),
            'daily_stats' => $this->getDailyStats($since),
        ];

        if ($request->wantsJson()) {
            return response()->json($analytics);
        }

        return view('admin.communication.analytics', compact('analytics', 'period'));
    }

    /**
     * Send bulk communication
     */
    public function bulkSend(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|in:email,sms',
            'channel' => 'required|in:all,email,sms',
            'template' => 'required|string',
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
            'recipients' => 'required|array',
            'recipients.*' => 'email',
        ]);

        // TODO: Implement bulk sending logic
        // This would involve queuing jobs for each recipient

        $batchId = uniqid('bulk_');

        foreach ($validated['recipients'] as $recipient) {
            $data = [
                'subject' => $validated['subject'],
                'message' => $validated['message'],
                'user' => auth()->user(),
                'batch_id' => $batchId,
            ];

            if ($validated['type'] === 'email') {
                $this->communicationService->sendEmail(
                    $validated['template'], 
                    $data, 
                    null, 
                    $recipient
                );
            } else {
                $this->communicationService->sendSms(
                    $validated['template'], 
                    $data, 
                    null, 
                    $recipient
                );
            }
        }

        return back()->with('success', "Bulk communication started for {$batchId}");
    }

    protected function getEmailProviderStatus(): array
    {
        $providers = config('communication.providers.email.options', []);
        $status = [];

        foreach ($providers as $provider) {
            try {
                $result = $this->emailService->testConfiguration($provider);
                $status[$provider] = $result;
            } catch (\Exception $e) {
                $status[$provider] = [
                    'success' => false,
                    'message' => $e->getMessage(),
                    'response_time' => 0,
                ];
            }
        }

        return $status;
    }

    protected function getSmsProviderStatus(): array
    {
        $providers = config('communication.providers.sms.options', []);
        $status = [];

        foreach ($providers as $provider) {
            try {
                $result = $this->smsService->testConfiguration($provider);
                $status[$provider] = $result;
            } catch (\Exception $e) {
                $status[$provider] = [
                    'success' => false,
                    'message' => $e->getMessage(),
                    'response_time' => 0,
                ];
            }
        }

        return $status;
    }

    protected function calculateSuccessRate($since): array
    {
        $total = DB::table('communication_logs')
            ->where('sent_at', '>=', $since)->count();
            
        $successful = DB::table('communication_logs')
            ->where('status', 'sent')
            ->where('sent_at', '>=', $since)->count();

        return [
            'percentage' => $total > 0 ? round(($successful / $total) * 100, 2) : 0,
            'total' => $total,
            'successful' => $successful,
            'failed' => $total - $successful,
        ];
    }

    protected function calculateCostAnalysis($since): array
    {
        return DB::table('communication_logs')
            ->select('channel', DB::raw('SUM(cost) as total_cost'), DB::raw('COUNT(*) as count'))
            ->where('sent_at', '>=', $since)
            ->whereNotNull('cost')
            ->groupBy('channel')
            ->get()
            ->keyBy('channel')
            ->toArray();
    }

    protected function getProviderPerformance($since): array
    {
        return DB::table('communication_logs')
            ->select('provider', 'channel', 
                DB::raw('COUNT(*) as total_sent'),
                DB::raw('SUM(CASE WHEN status = "sent" THEN 1 ELSE 0 END) as successful'),
                DB::raw('AVG(COST) as avg_cost'))
            ->where('sent_at', '>=', $since)
            ->groupBy('provider', 'channel')
            ->get()
            ->toArray();
    }

    protected function getDailyStats($since): array
    {
        return DB::table('communication_logs')
            ->select(
                DB::raw('DATE(sent_at) as date'),
                DB::raw('COUNT(*) as total'),
                DB::raw('SUM(CASE WHEN channel = "email" THEN 1 ELSE 0 END) as emails'),
                DB::raw('SUM(CASE WHEN channel = "sms" THEN 1 ELSE 0 END) as sms')
            )
            ->where('sent_at', '>=', $since)
            ->groupBy(DB::raw('DATE(sent_at)'))
            ->orderBy('date')
            ->get()
            ->toArray();
    }
}