<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Job;
use App\Models\Application;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
    }

    /**
     * Get platform statistics
     */
    public function stats(): JsonResponse
    {
        $stats = [
            'total_users' => User::count(),
            'active_jobs' => Job::where('status', 'active')->count(),
            'total_applications' => Application::count(),
            'revenue' => 0, // Placeholder for revenue calculation
            'new_users_today' => User::whereDate('created_at', today())->count(),
            'new_applications_today' => Application::whereDate('created_at', today())->count(),
            'pending_jobs' => Job::where('status', 'pending')->count(),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
            'message' => 'Statistics retrieved successfully'
        ]);
    }

    /**
     * Get all users
     */
    public function users(Request $request): JsonResponse
    {
        $query = User::with(['profile', 'company']);

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        $users = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $users,
            'message' => 'Users retrieved successfully'
        ]);
    }

    /**
     * Get all jobs
     */
    public function jobs(Request $request): JsonResponse
    {
        $query = Job::with(['company', 'applications']);

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('company', function ($companyQuery) use ($search) {
                      $companyQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $jobs = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $jobs,
            'message' => 'Jobs retrieved successfully'
        ]);
    }

    /**
     * Get all applications
     */
    public function applications(Request $request): JsonResponse
    {
        $query = Application::with(['user.profile', 'job.company']);

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        $applications = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $applications,
            'message' => 'Applications retrieved successfully'
        ]);
    }

    /**
     * Delete a user
     */
    public function deleteUser(User $user): JsonResponse
    {
        // Prevent deleting admin users
        if ($user->is_admin) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete admin user'
            ], 400);
        }

        // Delete related data
        if ($user->profile) {
            if ($user->profile->resume_path) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($user->profile->resume_path);
            }
            $user->profile->delete();
        }

        $user->applications()->delete();
        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully'
        ]);
    }

    /**
     * Delete a job
     */
    public function deleteJob(Job $job): JsonResponse
    {
        $job->delete();

        return response()->json([
            'success' => true,
            'message' => 'Job deleted successfully'
        ]);
    }

    /**
     * Update user status
     */
    public function updateUserStatus(Request $request, User $user): JsonResponse
    {
        $request->validate([
            'status' => 'required|in:active,suspended,banned',
        ]);

        $user->update(['status' => $request->status]);

        return response()->json([
            'success' => true,
            'data' => $user,
            'message' => 'User status updated successfully'
        ]);
    }

    /**
     * Update job status
     */
    public function updateJobStatus(Request $request, Job $job): JsonResponse
    {
        $request->validate([
            'status' => 'required|in:active,pending,rejected,expired',
        ]);

        $job->update(['status' => $request->status]);

        return response()->json([
            'success' => true,
            'data' => $job->load('company'),
            'message' => 'Job status updated successfully'
        ]);
    }
}