<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class CompanyController extends Controller
{
    /**
     * Get all companies
     */
    public function index(Request $request): JsonResponse
    {
        $query = Company::with(['user']);

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('sector', 'like', "%{$search}%");
            });
        }

        if ($request->has('sector') && $request->sector) {
            $query->where('sector', 'like', "%{$request->sector}%");
        }

        if ($request->has('location') && $request->location) {
            $query->where('location', 'like', "%{$request->location}%");
        }

        $companies = $query->orderBy('name')->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $companies,
            'message' => 'Companies retrieved successfully'
        ]);
    }

    /**
     * Get a specific company
     */
    public function show(Company $company): JsonResponse
    {
        $company->load(['user', 'jobs' => function ($query) {
            $query->where('status', 'active')->orderBy('created_at', 'desc');
        }]);

        return response()->json([
            'success' => true,
            'data' => $company,
            'message' => 'Company retrieved successfully'
        ]);
    }

    /**
     * Create a company profile (authenticated users only)
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:2000',
            'industry' => 'nullable|string|max:255',
            'location' => 'nullable|string|max:255',
            'website' => 'nullable|url',
            'size' => 'nullable|string|in:1-10,11-50,51-200,201-500,501-1000,1000+',
            'founded_year' => 'nullable|integer|min:1800|max:' . (date('Y') + 1),
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $user = auth()->user();

        // Check if user already has a company
        if ($user->company) {
            return response()->json([
                'success' => false,
                'message' => 'User already has a company profile'
            ], 400);
        }

        $companyData = [
            'owner_user_id' => $user->id,
            'name' => $request->name,
            'description' => $request->description,
            'sector' => $request->industry,
            'registration_no' => $request->registration_no,
        ];

        // Handle logo upload
        if ($request->hasFile('logo')) {
            $logoPath = $request->file('logo')->store('company-logos', 'public');
            $companyData['logo_path'] = $logoPath;
        }

        $company = Company::create($companyData);

        return response()->json([
            'success' => true,
            'data' => $company,
            'message' => 'Company profile created successfully'
        ], 201);
    }

    /**
     * Update company profile (company owner only)
     */
    public function update(Request $request, Company $company): JsonResponse
    {
        // Check if user owns this company
        if ($company->owner_user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:2000',
            'sector' => 'nullable|string|max:255',
            'registration_no' => 'nullable|string|max:255',
            'logo_path' => 'nullable|string|max:255',
        ]);

        $companyData = [
            'name' => $request->name,
            'description' => $request->description,
            'sector' => $request->sector,
            'registration_no' => $request->registration_no,
            'logo_path' => $request->logo_path,
        ];

        $company->update($companyData);

        return response()->json([
            'success' => true,
            'data' => $company,
            'message' => 'Company profile updated successfully'
        ]);
    }

    /**
     * Delete company profile (company owner only)
     */
    public function destroy(Company $company): JsonResponse
    {
        // Check if user owns this company
        if ($company->owner_user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        // Delete logo if exists
        if ($company->logo_path) {
            \Illuminate\Support\Facades\Storage::disk('public')->delete($company->logo_path);
        }

        $company->delete();

        return response()->json([
            'success' => true,
            'message' => 'Company profile deleted successfully'
        ]);
    }
}