<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Job;
use App\Models\Application;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Database\Eloquent\Builder;

class JobController extends Controller
{
    /**
     * Get all jobs with optional filtering
     */
    public function index(Request $request): JsonResponse
    {
        $query = Job::with(['company'])
            ->where('status', 'active')
            ->where(function ($q) {
                $q->where('application_deadline', '>', now())
                  ->orWhereNull('application_deadline');
            })
            ->orderBy('created_at', 'desc');

        // Advanced Search functionality
        if ($request->filled('q') || $request->filled('search')) {
            $searchTerm = $request->q ?? $request->search;
            $query->where(function (Builder $q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('skills', 'like', "%{$searchTerm}%")
                  ->orWhere('requirements', 'like', "%{$searchTerm}%")
                  ->orWhereHas('company', function (Builder $q) use ($searchTerm) {
                      $q->where('name', 'like', "%{$searchTerm}%")
                        ->orWhere('description', 'like', "%{$searchTerm}%");
                  });
            });
        }

        // Location filter
        if ($request->filled('location')) {
            if ($request->location === 'remote') {
                $query->where('remote_work', true);
            } else {
                $query->where('location', 'like', "%{$request->location}%");
            }
        }

        // Sector/Category filter
        if ($request->filled('sector')) {
            $query->where('sector', $request->sector);
        }

        // Experience level filter
        if ($request->filled('experience_level')) {
            $query->where('experience_level', $request->experience_level);
        }

        // Employment type filter
        if ($request->filled('employment_type')) {
            $query->where('employment_type', $request->employment_type);
        }

        // Salary range filters
        if ($request->filled('salary_min')) {
            $query->where('salary_max', '>=', $request->salary_min);
        }
        if ($request->filled('salary_max')) {
            $query->where('salary_min', '<=', $request->salary_max);
        }

        // Posted within filter
        if ($request->filled('posted_within')) {
            $days = (int) $request->posted_within;
            $query->where('created_at', '>=', now()->subDays($days));
        }

        // Remote jobs filter
        if ($request->boolean('remote')) {
            $query->where('remote_work', true);
        }

        // Featured jobs filter
        if ($request->boolean('featured')) {
            $query->where('featured', true);
        }

        // Jobs with salary info filter
        if ($request->boolean('with_salary')) {
            $query->whereNotNull('salary_min');
        }

        // Advanced Sorting
        $sortBy = $request->get('sort', 'newest');
        switch ($sortBy) {
            case 'oldest':
                $query->orderBy('created_at', 'asc');
                break;
            case 'salary_high':
                $query->orderByRaw('COALESCE(salary_max, salary_min) DESC');
                break;
            case 'salary_low':
                $query->orderByRaw('COALESCE(salary_min, salary_max) ASC');
                break;
            case 'relevance':
                // If there's a search term, prioritize relevance
                if ($request->filled('q') || $request->filled('search')) {
                    $searchTerm = $request->q ?? $request->search;
                    $query->orderByRaw("
                        CASE
                            WHEN title LIKE ? THEN 1
                            WHEN description LIKE ? THEN 2
                            WHEN skills LIKE ? THEN 3
                            ELSE 4
                        END
                    ", ["%{$searchTerm}%", "%{$searchTerm}%", "%{$searchTerm}%"])
                    ->orderBy('created_at', 'desc');
                } else {
                    $query->orderBy('created_at', 'desc');
                }
                break;
            case 'newest':
            default:
                $query->orderBy('created_at', 'desc');
                break;
        }

        $jobs = $query->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $jobs,
            'message' => 'Jobs retrieved successfully'
        ]);
    }

    /**
     * Get a specific job
     */
    public function show(Job $job): JsonResponse
    {
        if ($job->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Job not found'
            ], 404);
        }

        $job->load(['company', 'applications' => function ($query) {
            $query->where('user_id', auth()->id());
        }]);

        return response()->json([
            'success' => true,
            'data' => $job,
            'message' => 'Job retrieved successfully'
        ]);
    }

    /**
     * Search jobs by query
     */
    public function search(string $query): JsonResponse
    {
        $jobs = Job::with(['company'])
            ->where('status', 'active')
            ->where(function ($q) use ($query) {
                $q->where('title', 'like', "%{$query}%")
                  ->orWhere('description', 'like', "%{$query}%")
                  ->orWhere('skills', 'like', "%{$query}%")
                  ->orWhereHas('company', function ($companyQuery) use ($query) {
                      $companyQuery->where('name', 'like', "%{$query}%");
                  });
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $jobs,
            'message' => 'Search results retrieved successfully'
        ]);
    }

    /**
     * Create a new job (company only)
     */
    public function store(Request $request): JsonResponse
    {
        $this->authorize('create', Job::class);

        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'requirements' => 'nullable|string',
            'responsibilities' => 'nullable|string',
            'benefits' => 'nullable|string',
            'location' => 'required|string',
            'employment_type' => 'required|in:full-time,part-time,contract,freelance,internship',
            'experience_level' => 'required|in:entry,mid,senior,executive',
            'salary_min' => 'nullable|numeric|min:0',
            'salary_max' => 'nullable|numeric|min:0|gte:salary_min',
            'skills' => 'nullable|string',
            'application_deadline' => 'nullable|date|after:today',
            'featured' => 'boolean',
        ]);

        $user = auth()->user();
        $company = $user->company;

        $job = Job::create([
            'company_id' => $company->id,
            'title' => $request->title,
            'description' => $request->description,
            'requirements' => $request->requirements,
            'responsibilities' => $request->responsibilities,
            'benefits' => $request->benefits,
            'location' => $request->location,
            'employment_type' => $request->employment_type,
            'experience_level' => $request->experience_level,
            'salary_min' => $request->salary_min,
            'salary_max' => $request->salary_max,
            'skills' => $request->skills,
            'application_deadline' => $request->application_deadline,
            'featured' => $request->featured ?? false,
            'status' => 'active',
        ]);

        return response()->json([
            'success' => true,
            'data' => $job->load('company'),
            'message' => 'Job created successfully'
        ], 201);
    }

    /**
     * Update a job (company only)
     */
    public function update(Request $request, Job $job): JsonResponse
    {
        $this->authorize('update', $job);

        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'requirements' => 'nullable|string',
            'responsibilities' => 'nullable|string',
            'benefits' => 'nullable|string',
            'location' => 'required|string',
            'employment_type' => 'required|in:full-time,part-time,contract,freelance,internship',
            'experience_level' => 'required|in:entry,mid,senior,executive',
            'salary_min' => 'nullable|numeric|min:0',
            'salary_max' => 'nullable|numeric|min:0|gte:salary_min',
            'skills' => 'nullable|string',
            'application_deadline' => 'nullable|date|after:today',
            'featured' => 'boolean',
            'status' => 'required|in:active,pending,rejected,expired',
        ]);

        $job->update($request->all());

        return response()->json([
            'success' => true,
            'data' => $job->load('company'),
            'message' => 'Job updated successfully'
        ]);
    }

    /**
     * Delete a job (company only)
     */
    public function destroy(Job $job): JsonResponse
    {
        $this->authorize('delete', $job);

        $job->delete();

        return response()->json([
            'success' => true,
            'message' => 'Job deleted successfully'
        ]);
    }

    /**
     * Like or unlike a job
     */
    public function like(Request $request, Job $job): JsonResponse
    {
        $user = auth()->user();
        $existingLike = \App\Models\JobLike::where('user_id', $user->id)
            ->where('job_id', $job->id)
            ->first();

        if ($existingLike) {
            $existingLike->delete();
            $liked = false;
        } else {
            \App\Models\JobLike::create([
                'user_id' => $user->id,
                'job_id' => $job->id,
            ]);
            $liked = true;
        }

        return response()->json([
            'success' => true,
            'liked' => $liked,
            'message' => $liked ? 'Job liked successfully' : 'Job unliked successfully'
        ]);
    }

    /**
     * Get comments for a job
     */
    public function getComments(Job $job): JsonResponse
    {
        $comments = $job->comments()
            ->with(['user', 'replies.user'])
            ->topLevel()
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $comments,
            'message' => 'Comments retrieved successfully'
        ]);
    }

    /**
     * Add a comment to a job
     */
    public function addComment(Request $request, Job $job): JsonResponse
    {
        $request->validate([
            'comment' => 'required|string|max:1000',
            'parent_id' => 'nullable|exists:job_comments,id'
        ]);

        $user = auth()->user();

        $comment = \App\Models\JobComment::create([
            'user_id' => $user->id,
            'job_id' => $job->id,
            'comment' => $request->comment,
            'parent_id' => $request->parent_id,
        ]);

        $comment->load(['user', 'replies.user']);

        return response()->json([
            'success' => true,
            'data' => $comment,
            'message' => 'Comment added successfully'
        ]);
    }

    /**
     * Report a job
     */
    public function report(Request $request, Job $job): JsonResponse
    {
        $request->validate([
            'reason' => 'required|string|max:500'
        ]);

        $user = auth()->user();

        // Create a report record (you might want to create a JobReport model)
        // For now, we'll just log it
        \App\Services\UserActivityService::log('report_job', 'job', $job->id, 'Reported job: ' . $request->reason);

        return response()->json([
            'success' => true,
            'message' => 'Job reported successfully. We will review it shortly.'
        ]);
    }
}