<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class SubscriptionController extends Controller
{
    /**
     * Get current user's company subscription
     */
    public function show(): JsonResponse
    {
        $user = auth()->user();
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        $subscription = $company->subscription;

        return response()->json([
            'success' => true,
            'data' => $subscription,
            'message' => 'Subscription retrieved successfully'
        ]);
    }

    /**
     * Get available subscription plans
     */
    public function plans(): JsonResponse
    {
        $plans = [
            'basic' => [
                'name' => 'Basic',
                'price' => 5000,
                'currency' => 'XAF',
                'features' => [
                    '5 job posts per month',
                    'Basic support',
                    'Standard job visibility'
                ],
                'limits' => [
                    'job_posts' => 5,
                    'featured_jobs' => 0,
                    'ats_enabled' => false
                ]
            ],
            'pro' => [
                'name' => 'Pro',
                'price' => 15000,
                'currency' => 'XAF',
                'features' => [
                    '50 job posts per month',
                    'Priority support',
                    'Featured job listings',
                    'Advanced ATS features',
                    'Analytics dashboard'
                ],
                'limits' => [
                    'job_posts' => 50,
                    'featured_jobs' => 3,
                    'ats_enabled' => true
                ]
            ],
            'enterprise' => [
                'name' => 'Enterprise',
                'price' => 50000,
                'currency' => 'XAF',
                'features' => [
                    'Unlimited job posts',
                    'Dedicated support',
                    'Premium featured listings',
                    'Advanced ATS with AI',
                    'Custom integrations',
                    'White-label options'
                ],
                'limits' => [
                    'job_posts' => -1, // unlimited
                    'featured_jobs' => 10,
                    'ats_enabled' => true
                ]
            ]
        ];

        return response()->json([
            'success' => true,
            'data' => $plans,
            'message' => 'Subscription plans retrieved successfully'
        ]);
    }

    /**
     * Subscribe to a plan (initiates payment)
     */
    public function subscribe(Request $request): JsonResponse
    {
        $request->validate([
            'plan' => 'required|in:basic,pro,enterprise',
        ]);

        $user = auth()->user();
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        // Check if already has active subscription
        if ($company->subscription && $company->subscription->status === 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Active subscription already exists'
            ], 400);
        }

        $plans = [
            'basic' => 5000,
            'pro' => 15000,
            'enterprise' => 50000,
        ];

        $amount = $plans[$request->plan];

        // Record pending transaction
        $txId = \Illuminate\Support\Facades\DB::table('payment_transactions')->insertGetId([
            'company_id' => $company->id,
            'amount_cfa' => $amount,
            'currency' => 'XAF',
            'status' => 'pending',
            'purpose' => $request->plan . '_subscription',
            'meta' => json_encode(['plan' => $request->plan]),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // For now, return transaction ID - payment processing would be handled by frontend
        return response()->json([
            'success' => true,
            'data' => [
                'transaction_id' => $txId,
                'amount' => $amount,
                'plan' => $request->plan,
            ],
            'message' => 'Subscription initiated. Proceed to payment.'
        ]);
    }

    /**
     * Cancel subscription
     */
    public function cancel(): JsonResponse
    {
        $user = auth()->user();
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        $subscription = $company->subscription;

        if (!$subscription || $subscription->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'No active subscription to cancel'
            ], 400);
        }

        $subscription->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'data' => $subscription,
            'message' => 'Subscription cancelled successfully'
        ]);
    }

    /**
     * Renew subscription
     */
    public function renew(): JsonResponse
    {
        $user = auth()->user();
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        $subscription = $company->subscription;

        if (!$subscription) {
            return response()->json([
                'success' => false,
                'message' => 'No subscription found'
            ], 404);
        }

        // Calculate renewal amount based on plan
        $plans = [
            'Basic' => 5000,
            'Pro' => 15000,
            'Enterprise' => 50000,
        ];

        $amount = $plans[$subscription->plan_name] ?? 15000;

        // Record pending transaction
        $txId = \Illuminate\Support\Facades\DB::table('payment_transactions')->insertGetId([
            'company_id' => $company->id,
            'amount_cfa' => $amount,
            'currency' => 'XAF',
            'status' => 'pending',
            'purpose' => 'subscription_renewal',
            'meta' => json_encode(['plan' => $subscription->plan_name, 'renewal' => true]),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'data' => [
                'transaction_id' => $txId,
                'amount' => $amount,
                'plan' => $subscription->plan_name,
            ],
            'message' => 'Renewal initiated. Proceed to payment.'
        ]);
    }

    /**
     * Get subscription transactions
     */
    public function transactions(): JsonResponse
    {
        $user = auth()->user();
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        $transactions = \Illuminate\Support\Facades\DB::table('payment_transactions')
            ->where('company_id', $company->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $transactions,
            'message' => 'Transactions retrieved successfully'
        ]);
    }
}
