<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Jobs\HandleAdActivation;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\DB;
use App\Services\MesombPaymentService;

class PaymentWebhookController extends Controller
{
    private MesombPaymentService $mesombService;

    public function __construct(MesombPaymentService $mesombService)
    {
        $this->mesombService = $mesombService;
    }

    public function handle(Request $request): Response
    {
        try {
            // Get raw body for signature verification
            $rawBody = $request->getContent();
            $payload = $request->all();
            
            Log::info('Mesomb webhook received', [
                'payload' => $payload,
                'headers' => $request->headers->all(),
                'timestamp' => now()
            ]);

            // Verify webhook signature for security
            $signature = $request->header('X-MeSomb-Signature');
            $timestamp = $request->header('X-MeSomb-Timestamp');
            
            if (!$this->verifyWebhookSignature($signature, $timestamp, $rawBody)) {
                Log::warning('Invalid webhook signature', [
                    'signature' => $signature,
                    'timestamp' => $timestamp
                ]);
                return response()->noContent(Response::HTTP_UNAUTHORIZED);
            }

            // Extract payment information
            $paymentInfo = $this->extractPaymentInfo($payload);
            
            if (!$this->isValidPaymentInfo($paymentInfo)) {
                Log::warning('Invalid payment info received', ['payment_info' => $paymentInfo]);
                return response()->noContent(Response::HTTP_BAD_REQUEST);
            }

            // First try to find platform subscription/payment transaction
            $transaction = $this->findTransaction($paymentInfo);

            if ($transaction) {
                // Process platform/subscription payment
                $this->processPaymentUpdate($transaction, $paymentInfo);
                Log::info('Webhook processed for platform transaction', [
                    'transaction_id' => $transaction->id,
                    'payment_status' => $paymentInfo['status']
                ]);

                return response()->noContent();
            }

            // If not found, attempt to match an ad payment in ad_transactions
            $adTransaction = $this->findAdTransaction($paymentInfo);

            if (!$adTransaction) {
                Log::warning('Transaction not found in both payment_transactions and ad_transactions', [
                    'payment_id' => $paymentInfo['payment_id'],
                    'reference' => $paymentInfo['reference']
                ]);
                return response()->noContent(Response::HTTP_NOT_FOUND);
            }

            // Process ad payment update
            $this->processAdTransactionUpdate($adTransaction, $paymentInfo);

            Log::info('Webhook processed successfully', [
                'ad_transaction_id' => $adTransaction->id ?? null,
                'payment_status' => $paymentInfo['status']
            ]);

            return response()->noContent();

        } catch (\Exception $e) {
            Log::error('Webhook processing failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return response()->noContent(Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Find ad transaction by provider reference or id
     */
    private function findAdTransaction(array $paymentInfo)
    {
        return DB::table('ad_transactions')
            ->where(function ($query) use ($paymentInfo) {
                $query->where('provider_reference', $paymentInfo['reference'])
                      ->orWhere('id', $paymentInfo['payment_id']);
            })
            ->orderByDesc('id')
            ->first();
    }

    /**
     * Process updates for ad_transactions
     */
    private function processAdTransactionUpdate($adTransaction, array $paymentInfo): void
    {
        $status = $paymentInfo['status'];

            // idempotent update of transaction row
            $newStatus = $paymentInfo['status'] ?? 'unknown';
            DB::table('ad_transactions')->where('id', $adTransaction->id)->update([
                'status' => $newStatus,
                'raw_payload' => json_encode($paymentInfo),
                'updated_at' => now(),
            ]);

            if ($newStatus === 'SUCCESS') {
                $ad = DB::table('ads')->where('id', $adTransaction->ad_id)->first();
                if ($ad && $ad->status !== 'active') {
                    $plan = DB::table('ad_plans')->where('id', $ad->ad_plan_id)->first();
                    $start = now();
                    $end = null;
                    if ($plan) {
                        try {
                            $end = now()->addDays((int) $plan->days);
                        } catch (\Exception $e) {
                            Log::warning('Failed to compute end_at from plan', ['ad_id' => $ad->id, 'plan_id' => $plan->id]);
                        }
                    }

                    DB::table('ads')->where('id', $ad->id)->update([
                        'status' => 'active',
                        'start_at' => $start,
                        'end_at' => $end,
                        'updated_at' => now(),
                    ]);

                    // dispatch activation job
                    HandleAdActivation::dispatch($ad->id);
                }
            }

        // Update ad_transactions
        DB::table('ad_transactions')
            ->where('id', $adTransaction->id)
            ->update([
                'status' => $status,
                'raw_payload' => json_encode($paymentInfo),
                'updated_at' => now()
            ]);

        if (in_array($status, ['success','completed','succeeded'])) {
            // Activate associated ad if present
            if ($adTransaction->ad_id) {
                $ad = DB::table('ads')->where('id', $adTransaction->ad_id)->first();
                if ($ad) {
                    // If ad already active, ensure idempotency
                    if ($ad->status === 'active') {
                        Log::info('Ad already active', ['ad_id' => $ad->id]);
                        return;
                    }

                    $plan = DB::table('ad_plans')->where('id', $ad->ad_plan_id)->first();
                    $days = $plan->days ?? 0;

                    $startAt = $ad->start_at ? \Carbon\Carbon::parse($ad->start_at) : now();
                    $endAt = $startAt->copy()->addDays($days);

                    DB::table('ads')
                        ->where('id', $ad->id)
                        ->update([
                            'status' => ($startAt->greaterThan(now()) ? 'scheduled' : 'active'),
                            'start_at' => $startAt,
                            'end_at' => $endAt,
                            'updated_at' => now()
                        ]);

                    Log::info('Ad activated via webhook', ['ad_id' => $ad->id, 'start_at' => $startAt, 'end_at' => $endAt]);

                    // Send activation notification (non-blocking)
                    try {
                        $user = DB::table('users')->where('id', $ad->user_id)->first();
                        if ($user && $user->email) {
                            // For now log; place to dispatch Mailables / Notifications
                            Log::info('Would send AdActivated notification to', ['email' => $user->email, 'ad_id' => $ad->id]);
                        }
                    } catch (\Exception $e) {
                        Log::error('Failed to dispatch ad activation notification', ['error' => $e->getMessage(), 'ad_id' => $ad->id]);
                    }
                }
            }
        } else {
            Log::warning('Ad transaction not successful', ['ad_transaction_id' => $adTransaction->id, 'status' => $status]);
        }
    }

    /**
     * Verify webhook signature
     */
    private function verifyWebhookSignature(?string $signature, ?string $timestamp, string $rawBody): bool
    {
        if (!$signature || !$timestamp) {
            return false;
        }

        // Check if timestamp is recent (within 5 minutes)
        if (abs(time() - (int)$timestamp) > 300) {
            return false;
        }

        return $this->mesombService->verifyWebhookSignature($signature, $timestamp, $rawBody);
    }

    /**
     * Extract payment information from webhook payload
     */
    private function extractPaymentInfo(array $payload): array
    {
        return [
            'payment_id' => $payload['payment_id'] ?? null,
            'reference' => $payload['trxref'] ?? $payload['reference'] ?? null,
            'status' => strtolower($payload['status'] ?? ''),
            'amount' => $payload['amount'] ?? null,
            'currency' => $payload['currency'] ?? 'XAF',
            'operator' => $payload['operator'] ?? null,
            'phone' => $payload['phone'] ?? null,
            'message' => $payload['message'] ?? null,
            'created_at' => $payload['created'] ?? null,
            'fees' => $payload['fees'] ?? null,
        ];
    }

    /**
     * Validate payment information
     */
    private function isValidPaymentInfo(array $info): bool
    {
        return !empty($info['payment_id']) && 
               !empty($info['reference']) && 
               !empty($info['status']);
    }

    /**
     * Find transaction by payment ID or reference
     */
    private function findTransaction(array $paymentInfo)
    {
        return DB::table('payment_transactions')
            ->where(function ($query) use ($paymentInfo) {
                $query->where('payment_id', $paymentInfo['payment_id'])
                      ->orWhere('provider_ref', $paymentInfo['reference']);
            })
            ->orderByDesc('id')
            ->first();
    }

    /**
     * Process payment update based on status
     */
    private function processPaymentUpdate($transaction, array $paymentInfo): void
    {
        $status = $paymentInfo['status'];
        
        // Update transaction status
        DB::table('payment_transactions')
            ->where('id', $transaction->id)
            ->update([
                'status' => $status,
                'provider_response' => json_encode($paymentInfo),
                'updated_at' => now()
            ]);

        switch ($status) {
            case 'success':
            case 'completed':
            case 'pending':
                $this->handleSuccessfulPayment($transaction, $paymentInfo);
                break;
                
            case 'failed':
            case 'cancelled':
            case 'expired':
                $this->handleFailedPayment($transaction, $paymentInfo);
                break;
                
            default:
                Log::info('Unhandled payment status', [
                    'status' => $status,
                    'transaction_id' => $transaction->id
                ]);
        }
    }

    /**
     * Handle successful payment
     */
    private function handleSuccessfulPayment($transaction, array $paymentInfo): void
    {
        // Parse transaction metadata
        $metadata = json_decode($transaction->metadata, true) ?: [];
        $plan = $metadata['plan'] ?? 'pro';
        $isRenewal = $metadata['renewal'] ?? false;

        // Define plan configurations
        $planConfigs = [
            'basic' => [
                'job_post_limit' => 5,
                'ats_enabled' => false,
                'featured_slots' => 0,
                'duration_days' => 30,
            ],
            'pro' => [
                'job_post_limit' => 50,
                'ats_enabled' => true,
                'featured_slots' => 3,
                'duration_days' => 30,
            ],
            'enterprise' => [
                'job_post_limit' => -1, // unlimited
                'ats_enabled' => true,
                'featured_slots' => 10,
                'duration_days' => 30,
            ],
        ];

        $config = $planConfigs[$plan] ?? $planConfigs['pro'];
        $durationDays = $config['duration_days'];

        if ($paymentInfo['status'] === 'pending') {
            // Payment is pending, don't activate subscription yet
            Log::info('Payment pending - subscription not activated', [
                'transaction_id' => $transaction->id,
                'plan' => $plan
            ]);
            return;
        }

        $this->activateOrExtendSubscription($transaction, $plan, $config, $isRenewal, $durationDays);

        // Send confirmation notification
        $this->sendPaymentConfirmationNotification($transaction, $paymentInfo);

        Log::info('Payment processed successfully', [
            'transaction_id' => $transaction->id,
            'plan' => $plan,
            'renewal' => $isRenewal
        ]);
    }

    /**
     * Handle failed payment
     */
    private function handleFailedPayment($transaction, array $paymentInfo): void
    {
        // Log failed payment for analysis
        Log::warning('Payment failed', [
            'transaction_id' => $transaction->id,
            'company_id' => $transaction->company_id,
            'amount' => $transaction->amount_cfa,
            'purpose' => $transaction->purpose,
            'status' => $paymentInfo['status'],
            'message' => $paymentInfo['message'] ?? null
        ]);

        // Don't deactivate existing subscription for failed payments
        // Just log the failure for retry logic if needed
    }

    /**
     * Activate or extend company subscription
     */
    private function activateOrExtendSubscription($transaction, string $plan, array $config, bool $isRenewal, int $durationDays): void
    {
        $now = now();
        $endsAt = $now->copy()->addDays($durationDays);

        if ($isRenewal) {
            // For renewal, extend existing subscription
            $existingSubscription = DB::table('subscriptions')
                ->where('company_id', $transaction->company_id)
                ->where('status', 'active')
                ->first();

            if ($existingSubscription) {
                $currentEnd = \Carbon\Carbon::parse($existingSubscription->ends_at);
                $newEnd = $currentEnd->greaterThan($now) 
                    ? $currentEnd->addDays($durationDays) 
                    : $endsAt;

                DB::table('subscriptions')
                    ->where('id', $existingSubscription->id)
                    ->update([
                        'ends_at' => $newEnd,
                        'updated_at' => $now,
                    ]);

                Log::info('Subscription extended', [
                    'subscription_id' => $existingSubscription->id,
                    'company_id' => $transaction->company_id,
                    'new_end_date' => $newEnd
                ]);
            } else {
                // No active subscription, create new one
                $this->createNewSubscription($transaction, $plan, $config, $now, $endsAt);
            }
        } else {
            // New subscription
            $this->createNewSubscription($transaction, $plan, $config, $now, $endsAt);
        }
    }

    /**
     * Create new subscription
     */
    private function createNewSubscription($transaction, string $plan, array $config, $now, $endsAt): void
    {
        DB::table('subscriptions')->updateOrInsert(
            ['company_id' => $transaction->company_id],
            [
                'plan_name' => ucfirst($plan),
                'starts_at' => $now,
                'ends_at' => $endsAt,
                'status' => 'active',
                'job_post_limit' => $config['job_post_limit'],
                'ats_enabled' => $config['ats_enabled'],
                'featured_slots' => $config['featured_slots'],
                'updated_at' => $now,
                'created_at' => $now,
            ]
        );

        Log::info('New subscription created', [
            'company_id' => $transaction->company_id,
            'plan' => $plan,
            'starts_at' => $now,
            'ends_at' => $endsAt
        ]);
    }

    /**
     * Send payment confirmation notification
     */
    private function sendPaymentConfirmationNotification($transaction, array $paymentInfo): void
    {
        try {
            $company = DB::table('companies')
                ->where('id', $transaction->company_id)
                ->first();

            if ($company && $company->email) {
                // Here you would typically use Laravel's notification system
                // to send an email notification
                
                Log::info('Payment confirmation would be sent', [
                    'company_email' => $company->email,
                    'company_name' => $company->name,
                    'amount' => $paymentInfo['amount']
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Failed to send payment confirmation', [
                'error' => $e->getMessage(),
                'transaction_id' => $transaction->id
            ]);
        }
    }
}
