<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Response;
use Illuminate\Validation\Rule;
use App\Models\User;

class ProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function edit(Request $request)
    {
        return view('profile.edit');
    }

    /**
     * Update the user's profile information.
     */
    public function update(Request $request)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('users')->ignore(auth()->id())],
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date|before:today',
            'location' => 'nullable|string|max:255',
            'bio' => 'nullable|string|max:1000',
        ]);

        $user = auth()->user();

        // Update name
        $user->name = $request->first_name . ' ' . $request->last_name;
        $user->email = $request->email;
        $user->save();

        // Update or create profile
        $user->profile()->updateOrCreate(
            ['user_id' => $user->id],
            [
                'phone' => $request->phone,
                'date_of_birth' => $request->date_of_birth,
                'location' => $request->location,
                'bio' => $request->bio,
            ]
        );

        return back()->with('success', 'Profile updated successfully.');
    }

    /**
     * Update professional information.
     */
    public function updateProfessional(Request $request)
    {
        // Only allow job seekers to update professional info
        /** @var User $authUser */
        $authUser = auth()->user();
        if ($authUser->isCompany()) {
            abort(403, 'Company accounts cannot update professional information.');
        }

        $request->validate([
            'current_job_title' => 'nullable|string|max:255',
            'experience_years' => 'nullable|string',
            'skills' => 'nullable|string|max:1000',
            'education_level' => 'nullable|string',
            'expected_salary' => 'nullable|numeric|min:0',
            'availability' => 'nullable|string',
            'resume' => 'nullable|file|mimes:pdf,doc,docx|max:5120',
        ]);

        $user = auth()->user();

        $profileData = [
            'current_job_title' => $request->current_job_title,
            'experience_years' => $request->experience_years,
            'skills' => $request->skills,
            'education_level' => $request->education_level,
            'expected_salary' => $request->expected_salary,
            'availability' => $request->availability,
        ];

        // Handle resume upload
        if ($request->hasFile('resume')) {
            // Delete old resume if exists
            if ($user->profile && $user->profile->resume_path) {
                Storage::disk('public')->delete($user->profile->resume_path);
            }

            $resumePath = $request->file('resume')->store('resumes', 'public');
            $profileData['resume_path'] = $resumePath;
        }

        $user->profile()->updateOrCreate(
            ['user_id' => $user->id],
            $profileData
        );

        return back()->with('success', 'Professional information updated successfully.');
    }

    /**
     * Update company information.
     */
    public function updateCompany(Request $request)
    {
        // Only allow company accounts to update company info
        /** @var User $authUser */
        $authUser = auth()->user();
        if (!$authUser->isCompany()) {
            abort(403, 'Personal accounts cannot update company information.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:2000',
            'industry' => 'nullable|string|max:255',
            'size' => 'nullable|string|max:50',
            'founded_year' => 'nullable|integer|min:1800|max:' . date('Y'),
            'location' => 'nullable|string|max:255',
            'website' => 'nullable|url|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        /** @var User $user */
        $user = auth()->user();

        $companyData = [
            'name' => $request->name,
            'description' => $request->description,
            'industry' => $request->industry,
            'size' => $request->size,
            'founded_year' => $request->founded_year,
            'location' => $request->location,
            'website' => $request->website,
        ];

        // Handle logo upload
        if ($request->hasFile('logo')) {
            // Delete old logo if exists
            if ($user->company && $user->company->logo) {
                Storage::disk('public')->delete($user->company->logo);
            }

            $logoPath = $request->file('logo')->store('company-logos', 'public');
            $companyData['logo'] = $logoPath;
        }

        $user->company()->updateOrCreate(
            ['owner_id' => $user->id],
            $companyData
        );

        return back()->with('success', 'Company information updated successfully.');
    }

    /**
     * Update user preferences.
     */
    public function updatePreferences(Request $request)
    {
        $request->validate([
            'remote_preference' => 'boolean',
            'email_notifications' => 'boolean',
            'preferred_industries' => 'nullable|array',
        ]);

        $user = auth()->user();

        $user->profile()->updateOrCreate(
            ['user_id' => $user->id],
            [
                'remote_preference' => $request->boolean('remote_preference'),
                'email_notifications' => $request->boolean('email_notifications'),
                'preferred_industries' => $request->preferred_industries ? json_encode($request->preferred_industries) : null,
            ]
        );

        return back()->with('success', 'Preferences updated successfully.');
    }

    /**
     * Update the user's password.
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|min:8|confirmed',
        ]);

        $user = auth()->user();

        // Check current password
        if (!Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'The current password is incorrect.']);
        }

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return back()->with('success', 'Password updated successfully.');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request)
    {
        $request->validate([
            'password' => 'required',
        ]);

        $user = auth()->user();

        // Check password
        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors(['password' => 'The password is incorrect.']);
        }

        // Delete profile and related data
        if ($user->profile) {
            if ($user->profile->resume_path) {
                Storage::disk('public')->delete($user->profile->resume_path);
            }
            $user->profile->delete();
        }

        // Delete applications
        $user->applications()->delete();

        // Delete user
        $user->delete();

        return redirect('/')->with('success', 'Account deleted successfully.');
    }

    /**
     * Download user's resume.
     */
    public function downloadResume()
    {
        $user = auth()->user();

        if (!$user->profile || !$user->profile->resume_path) {
            abort(404, 'Resume not found.');
        }

        return Response::download(storage_path('app/public/' . $user->profile->resume_path));
    }
}
