<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Cache\RateLimiter;
use Illuminate\Http\Exceptions\ThrottleRequestsException;
use Symfony\Component\HttpFoundation\Response;

class ApiRateLimit
{
    public function handle(Request $request, Closure $next, string $limit = '60', string $period = '1')
    {
        $rateLimiter = app(RateLimiter::class);
        
        $key = $request->user()
            ? 'user_' . $request->user()->id
            : 'ip_' . $request->ip();

        if ($rateLimiter->tooManyAttempts($key, (int) $limit, $period)) {
            $retryAfter = $rateLimiter->availableIn($key);
            
            return response()->json([
                'error' => 'Rate limit exceeded',
                'message' => sprintf('API rate limit exceeded. Maximum %s requests per %s minutes.', $limit, $period),
                'retry_after' => $retryAfter,
            ], Response::HTTP_TOO_MANY_REQUESTS, [
                'X-RateLimit-Limit' => (string) $limit,
                'X-RateLimit-Remaining' => '0',
                'X-RateLimit-Reset' => (string) now()->addSeconds($retryAfter)->timestamp,
            ]);
        }

        $response = $next($request);

        $rateLimiter->hit($key, $period * 60);

        // Add rate limit headers
        $remaining = $limit - $rateLimiter->attempts($key);
        $response->headers->set('X-RateLimit-Limit', (string) $limit);
        $response->headers->set('X-RateLimit-Remaining', (string) max(0, $remaining));
        $response->headers->set('X-RateLimit-Reset', (string) now()->addMinutes($period)->timestamp);

        return $response;
    }
}