<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Notifications\AdActivated;

class HandleAdActivation implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $adId;

    /**
     * Create a new job instance.
     */
    public function __construct(int $adId)
    {
        $this->adId = $adId;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $ad = DB::table('ads')->where('id', $this->adId)->first();
            if (!$ad) {
                Log::warning('HandleAdActivation: ad not found', ['ad_id' => $this->adId]);
                return;
            }

            // Send notification to owner (email + in-app)
            $user = DB::table('users')->where('id', $ad->user_id)->first();
            if ($user) {
                // Using Notification facade via DB (we avoid loading the full User model in job context for safety)
                try {
                    // If the User model exists and uses Notifiable, you could do: $userModel->notify(new AdActivated($ad));
                    // Here we'll insert a simple database notification record if notifications table exists
                    if (config('database.notifications_table', 'notifications')) {
                        DB::table('notifications')->insert([
                            'id' => (string) \Illuminate\Support\Str::uuid(),
                            'type' => AdActivated::class,
                            'notifiable_type' => 'App\\Models\\User',
                            'notifiable_id' => $user->id,
                            'data' => json_encode(['ad_id' => $ad->id, 'message' => 'Your ad is now active']),
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }

                    // Send email via Notification if User model loaded
                    try {
                        if (class_exists('App\\Models\\User')) {
                            $userModel = (\App\Models\User::find($user->id));
                            if ($userModel) {
                                $userModel->notify(new AdActivated($ad));
                            }
                        }
                    } catch (\Exception $e) {
                        Log::error('Failed to send AdActivated notification via model', ['error' => $e->getMessage(), 'ad_id' => $ad->id]);
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to queue notification record', ['error' => $e->getMessage(), 'ad_id' => $ad->id]);
                }
            }

            // Optionally warm cache or fire analytic events here
            Log::info('Ad activation job completed', ['ad_id' => $ad->id]);
        } catch (\Exception $e) {
            Log::error('HandleAdActivation failed', ['error' => $e->getMessage(), 'ad_id' => $this->adId]);
            throw $e; // allow retry
        }
    }
}
