<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class ApiKey extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'key',
        'secret',
        'user_id',
        'permissions',
        'last_used_at',
        'requests_count',
        'rate_limit',
        'expires_at',
        'is_active',
        'description',
        'metadata',
    ];

    protected $hidden = [
        'secret',
    ];

    protected $casts = [
        'permissions' => 'array',
        'metadata' => 'array',
        'last_used_at' => 'datetime',
        'expires_at' => 'datetime',
        'is_active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true)
            ->where(function ($q) {
                $q->whereNull('expires_at')
                  ->orWhere('expires_at', '>', now());
            });
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function isExpired(): bool
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    public function canMakeRequest(): bool
    {
        if (!$this->is_active || $this->isExpired()) {
            return false;
        }

        // Check rate limit (simplified - in production you'd use Redis/cache)
        $hourStart = now()->startOfHour();
        $requestsThisHour = $this->requests_count; // This would be tracked separately

        return $requestsThisHour < $this->rate_limit;
    }

    public function hasPermission(string $permission): bool
    {
        if (empty($this->permissions)) {
            return false;
        }

        return in_array($permission, $this->permissions) ||
               in_array('*', $this->permissions);
    }

    public function recordUsage()
    {
        $this->update([
            'last_used_at' => now(),
            'requests_count' => $this->requests_count + 1,
        ]);
    }

    public static function generateKey(): string
    {
        return 'ak_' . Str::random(32);
    }

    public static function generateSecret(): string
    {
        return 'sk_' . Str::random(64);
    }

    public function getMaskedSecretAttribute(): string
    {
        return substr($this->secret, 0, 8) . '...' . substr($this->secret, -4);
    }
}
