<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Application extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'applications_optimized';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'job_id',
        'user_id',
        'status',
        'cover_letter',
        'resume_path',
        'portfolio_url',
        'salary_expectation',
        'available_start_date',
        'application_data',
        'interview_scheduled_at',
        'notes',
        'rating',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'available_start_date' => 'date',
        'interview_scheduled_at' => 'datetime',
        'application_data' => 'array',
    ];

    /**
     * Get the job that the application is for.
     */
    public function job()
    {
        return $this->belongsTo(Job::class, 'job_id');
    }

    /**
     * Get the user who made the application.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the organization through the job.
     */
    public function organization()
    {
        return $this->hasOneThrough(Organization::class, Job::class, 'id', 'id', 'job_id', 'organization_id');
    }

    /**
     * Get the interactions for the application.
     */
    public function interactions()
    {
        return $this->morphMany(Interaction::class, 'target');
    }

    /**
     * Check if application is pending.
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if application is reviewed.
     */
    public function isReviewed(): bool
    {
        return $this->status === 'reviewed';
    }

    /**
     * Check if application is in interview stage.
     */
    public function isInterview(): bool
    {
        return $this->status === 'interview';
    }

    /**
     * Check if application is hired.
     */
    public function isHired(): bool
    {
        return $this->status === 'hired';
    }

    /**
     * Check if application is rejected.
     */
    public function isRejected(): bool
    {
        return $this->status === 'rejected';
    }

    /**
     * Get status color for UI.
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            'pending' => 'yellow',
            'reviewed' => 'blue',
            'interview' => 'purple',
            'hired' => 'green',
            'rejected' => 'red',
            default => 'gray',
        };
    }

    /**
     * Get status label for UI.
     */
    public function getStatusLabelAttribute(): string
    {
        return match($this->status) {
            'pending' => 'Pending Review',
            'reviewed' => 'Under Review',
            'interview' => 'Interview Scheduled',
            'hired' => 'Hired',
            'rejected' => 'Rejected',
            default => 'Unknown',
        };
    }

    /**
     * Scope for applications by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for applications by job.
     */
    public function scopeForJob($query, $jobId)
    {
        return $query->where('job_id', $jobId);
    }

    /**
     * Scope for applications by user.
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }
}