<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class EmailTemplate extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'subject',
        'content',
        'variables',
        'type',
        'is_active',
    ];

    protected $casts = [
        'variables' => 'array',
        'is_active' => 'boolean',
    ];

    /**
     * Scope a query to only include active templates.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to filter by type.
     */
    public function scopeByType($query, string $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Render the template with variables.
     */
    public function render(array $variables = []): array
    {
        $subject = $this->subject;
        $content = $this->content;

        foreach ($variables as $key => $value) {
            $placeholder = '{{' . $key . '}}';
            $subject = str_replace($placeholder, $value, $subject);
            $content = str_replace($placeholder, $value, $content);
        }

        return [
            'subject' => $subject,
            'content' => $content,
        ];
    }

    /**
     * Get available variable placeholders.
     */
    public function getAvailableVariables(): array
    {
        if (empty($this->variables)) {
            return [];
        }

        return array_map(function ($var) {
            return '{{' . $var . '}}';
        }, $this->variables);
    }

    /**
     * Get the type display name.
     */
    public function getTypeDisplayAttribute(): string
    {
        return match($this->type) {
            'welcome' => 'Welcome Email',
            'notification' => 'Notification',
            'marketing' => 'Marketing',
            'system' => 'System',
            default => ucfirst($this->type),
        };
    }
}
