<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Interaction extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'interactions_optimized';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'target_type',
        'target_id',
        'interaction_type',
        'content',
        'rating',
        'parent_id',
        'status',
        'moderation_notes',
        'metadata',
        'interaction_data',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'metadata' => 'array',
        'interaction_data' => 'array',
    ];

    /**
     * Get the user who performed the interaction.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the parent interaction (for nested comments/replies).
     */
    public function parent()
    {
        return $this->belongsTo(Interaction::class, 'parent_id');
    }

    /**
     * Get the child interactions (replies).
     */
    public function replies()
    {
        return $this->hasMany(Interaction::class, 'parent_id');
    }

    /**
     * Get the target model (polymorphic relationship).
     */
    public function target()
    {
        return $this->morphTo();
    }

    /**
     * Check if interaction is active.
     */
    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Check if interaction is a like.
     */
    public function isLike(): bool
    {
        return $this->interaction_type === 'like';
    }

    /**
     * Check if interaction is a comment.
     */
    public function isComment(): bool
    {
        return $this->interaction_type === 'comment';
    }

    /**
     * Check if interaction is a review.
     */
    public function isReview(): bool
    {
        return $this->interaction_type === 'review';
    }

    /**
     * Check if interaction is a rating.
     */
    public function isRating(): bool
    {
        return $this->interaction_type === 'rating';
    }

    /**
     * Get interaction type label.
     */
    public function getTypeLabelAttribute(): string
    {
        return match($this->interaction_type) {
            'like' => 'Like',
            'unlike' => 'Unlike',
            'comment' => 'Comment',
            'reply' => 'Reply',
            'share' => 'Share',
            'view' => 'View',
            'report' => 'Report',
            'review' => 'Review',
            'rating' => 'Rating',
            'follow' => 'Follow',
            'bookmark' => 'Bookmark',
            default => ucfirst($this->interaction_type),
        };
    }

    /**
     * Scope for active interactions.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope for interactions by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('interaction_type', $type);
    }

    /**
     * Scope for interactions by target.
     */
    public function scopeForTarget($query, $type, $id)
    {
        return $query->where('target_type', $type)->where('target_id', $id);
    }

    /**
     * Scope for interactions by user.
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope for top-level interactions (no parent).
     */
    public function scopeTopLevel($query)
    {
        return $query->whereNull('parent_id');
    }
}