<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Transaction extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'transactions_optimized';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'organization_id',
        'type',
        'amount',
        'currency',
        'status',
        'provider',
        'provider_transaction_id',
        'provider_data',
        'subscription_plan',
        'billing_period_start',
        'billing_period_end',
        'plan_features',
        'api_key_id',
        'api_usage_quota',
        'api_usage_current',
        'api_usage_reset_date',
        'api_usage_details',
        'metadata',
        'description',
        'notes',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'amount' => 'decimal:2',
        'billing_period_start' => 'datetime',
        'billing_period_end' => 'datetime',
        'api_usage_reset_date' => 'datetime',
        'provider_data' => 'array',
        'plan_features' => 'array',
        'api_usage_details' => 'array',
        'metadata' => 'array',
    ];

    /**
     * Get the user associated with the transaction.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the organization associated with the transaction.
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class, 'organization_id');
    }

    /**
     * Check if transaction is pending.
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if transaction is completed.
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if transaction is failed.
     */
    public function isFailed(): bool
    {
        return $this->status === 'failed';
    }

    /**
     * Check if transaction is refunded.
     */
    public function isRefunded(): bool
    {
        return $this->status === 'refunded';
    }

    /**
     * Check if transaction is a subscription payment.
     */
    public function isSubscriptionPayment(): bool
    {
        return $this->type === 'subscription_payment';
    }

    /**
     * Check if transaction is a job payment.
     */
    public function isJobPayment(): bool
    {
        return $this->type === 'job_payment';
    }

    /**
     * Check if transaction is an API usage transaction.
     */
    public function isApiUsage(): bool
    {
        return $this->type === 'api_usage';
    }

    /**
     * Get transaction type label.
     */
    public function getTypeLabelAttribute(): string
    {
        return match($this->type) {
            'subscription_payment' => 'Subscription Payment',
            'job_payment' => 'Job Payment',
            'refund' => 'Refund',
            'api_usage' => 'API Usage',
            'premium_feature' => 'Premium Feature',
            'advertisement' => 'Advertisement',
            'bulk_email' => 'Bulk Email',
            'custom' => 'Custom',
            default => ucfirst(str_replace('_', ' ', $this->type)),
        };
    }

    /**
     * Get status color for UI.
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            'pending' => 'yellow',
            'completed' => 'green',
            'failed' => 'red',
            'cancelled' => 'gray',
            'refunded' => 'blue',
            default => 'gray',
        };
    }

    /**
     * Get formatted amount with currency.
     */
    public function getFormattedAmountAttribute(): string
    {
        return number_format($this->amount, 2) . ' ' . $this->currency;
    }

    /**
     * Scope for transactions by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for transactions by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope for transactions by user.
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope for transactions by organization.
     */
    public function scopeByOrganization($query, $organizationId)
    {
        return $query->where('organization_id', $organizationId);
    }

    /**
     * Scope for completed transactions.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }
}