<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\Application;

class ApplicationStatusUpdated extends Notification implements ShouldQueue
{
    use Queueable;

    public Application $application;
    public string $oldStatus;

    /**
     * Create a new notification instance.
     *
     * @param Application $application
     * @param string $oldStatus
     * @return void
     */
    public function __construct(Application $application, string $oldStatus)
    {
        $this->application = $application;
        $this->oldStatus = $oldStatus;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $application = $this->application;
        $job = $application->job;
        $company = $job->company;

        $statusLabels = [
            'pending' => 'Pending Review',
            'reviewing' => 'Under Review',
            'shortlisted' => 'Shortlisted',
            'interviewed' => 'Interview Scheduled',
            'offered' => 'Job Offered',
            'rejected' => 'Not Selected',
            'withdrawn' => 'Withdrawn',
        ];

        $newStatusLabel = $statusLabels[$application->status] ?? ucfirst($application->status);
        $oldStatusLabel = $statusLabels[$this->oldStatus] ?? ucfirst($this->oldStatus);

        return (new MailMessage)
            ->subject("Application Status Update: {$job->title}")
            ->greeting("Hello {$application->user->name}!")
            ->line("Your application status for **{$job->title}** at **{$company->name}** has been updated.")
            ->line("**Previous Status:** {$oldStatusLabel}")
            ->line("**New Status:** {$newStatusLabel}")
            ->when($application->review_notes, function ($mail) use ($application) {
                return $mail->line("**Notes from {$application->job->company->name}:** {$application->review_notes}");
            })
            ->when($application->status === 'interviewed', function ($mail) {
                return $mail->action('Prepare for Interview', route('applications.show', $this->application));
            })
            ->when($application->status === 'offered', function ($mail) {
                return $mail->action('View Offer Details', route('applications.show', $this->application));
            })
            ->when(in_array($application->status, ['rejected', 'withdrawn']), function ($mail) {
                return $mail->action('Browse More Jobs', route('jobs.index'));
            })
            ->line('Thank you for using Hire Cameroon!')
            ->salutation('Best regards, Hire Cameroon Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'application_id' => $this->application->id,
            'job_title' => $this->application->job->title,
            'company_name' => $this->application->job->company->name,
            'old_status' => $this->oldStatus,
            'new_status' => $this->application->status,
            'review_notes' => $this->application->review_notes,
        ];
    }
}
