<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\Subscription;

class SubscriptionActivated extends Notification implements ShouldQueue
{
    use Queueable;

    public Subscription $subscription;

    /**
     * Create a new notification instance.
     *
     * @param Subscription $subscription
     * @return void
     */
    public function __construct(Subscription $subscription)
    {
        $this->subscription = $subscription;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $subscription = $this->subscription;
        $company = $subscription->company;

        return (new MailMessage)
            ->subject("Welcome to Hire Cameroon {$subscription->plan_name} Plan!")
            ->greeting("Congratulations {$company->name}!")
            ->line("Your {$subscription->plan_name} subscription has been successfully activated.")
            ->line("**Subscription Details:**")
            ->line("• Plan: {$subscription->plan_name}")
            ->line("• Job Posts: " . ($subscription->job_post_limit === -1 ? 'Unlimited' : $subscription->job_post_limit))
            ->line("• Featured Jobs: {$subscription->featured_slots}")
            ->line("• ATS Features: " . ($subscription->ats_enabled ? 'Enabled' : 'Disabled'))
            ->line("• Valid until: {$subscription->ends_at->format('F j, Y')}")
            ->action('Start Posting Jobs', route('dashboard.jobs.create'))
            ->line('Thank you for choosing Hire Cameroon to find the best talent!')
            ->salutation('Best regards, Hire Cameroon Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'subscription_id' => $this->subscription->id,
            'plan_name' => $this->subscription->plan_name,
            'ends_at' => $this->subscription->ends_at,
            'job_post_limit' => $this->subscription->job_post_limit,
            'featured_slots' => $this->subscription->featured_slots,
            'ats_enabled' => $this->subscription->ats_enabled,
        ];
    }
}
