<?php

namespace App\Policies;

use App\Models\Application;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class ApplicationPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any applications.
     */
    public function viewAny(User $user)
    {
        return $user->hasRole(['admin', 'employer', 'recruiter']);
    }

    /**
     * Determine whether the user can view the application.
     */
    public function view(User $user, Application $application)
    {
        // Admin can view any application
        if ($user->hasRole('admin')) {
            return true;
        }

        // Job seeker can view their own applications
        if ($user->hasRole('job_seeker') && $application->user_id === $user->id) {
            return true;
        }

        // Company owner can view applications for their jobs
        if ($user->hasRole('employer') && $user->company) {
            return $application->job->company_id === $user->company->id;
        }

        // Recruiter can view applications if they have permission
        if ($user->hasRole('recruiter')) {
            return $user->hasPermissionTo('view_applicants');
        }

        return false;
    }

    /**
     * Determine whether the user can create applications.
     */
    public function create(User $user)
    {
        return $user->hasRole('job_seeker');
    }

    /**
     * Determine whether the user can update the application.
     */
    public function update(User $user, Application $application)
    {
        // Job seeker can update their own applications (withdraw, etc.)
        if ($user->hasRole('job_seeker') && $application->user_id === $user->id) {
            return true;
        }

        // Company can update status of applications for their jobs
        if ($user->hasRole(['admin', 'employer', 'recruiter']) && $user->company) {
            return $application->job->company_id === $user->company->id;
        }

        return false;
    }

    /**
     * Determine whether the user can delete the application.
     */
    public function delete(User $user, Application $application)
    {
        // Admin can delete any application
        if ($user->hasRole('admin')) {
            return true;
        }

        // Job seeker can delete their own applications
        if ($user->hasRole('job_seeker') && $application->user_id === $user->id) {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can manage application status.
     */
    public function manageStatus(User $user, Application $application)
    {
        // Admin can manage any application status
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can manage applications for their jobs
        if ($user->hasRole('employer') && $user->company) {
            return $application->job->company_id === $user->company->id;
        }

        // Recruiter can manage applications if they have permission
        if ($user->hasRole('recruiter')) {
            return $user->hasPermissionTo('manage_applicants');
        }

        return false;
    }
}