<?php

namespace App\Policies;

use App\Models\Job;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class JobPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any jobs.
     */
    public function viewAny(User $user)
    {
        return true; // Anyone can view jobs
    }

    /**
     * Determine whether the user can view the job.
     */
    public function view(User $user, Job $job)
    {
        return true; // Anyone can view active jobs
    }

    /**
     * Determine whether the user can create jobs.
     */
    public function create(User $user)
    {
        return $user->hasRole(['admin', 'employer']) && $user->company;
    }

    /**
     * Determine whether the user can update the job.
     */
    public function update(User $user, Job $job)
    {
        // Admin can update any job
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can update their own jobs
        return $user->hasRole('employer') &&
               $user->company &&
               $job->company_id === $user->company->id;
    }

    /**
     * Determine whether the user can delete the job.
     */
    public function delete(User $user, Job $job)
    {
        // Admin can delete any job
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can delete their own jobs
        return $user->hasRole('employer') &&
               $user->company &&
               $job->company_id === $user->company->id;
    }

    /**
     * Determine whether the user can restore the job.
     */
    public function restore(User $user, Job $job)
    {
        return $user->hasRole('admin');
    }

    /**
     * Determine whether the user can permanently delete the job.
     */
    public function forceDelete(User $user, Job $job)
    {
        return $user->hasRole('admin');
    }

    /**
     * Determine whether the user can manage applications for the job.
     */
    public function manageApplications(User $user, Job $job)
    {
        // Admin can manage any job's applications
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can manage their own job's applications
        return $user->hasRole('employer') &&
               $user->company &&
               $job->company_id === $user->company->id;
    }
}