<?php

namespace App\Services;

use App\Models\Ad;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class AdMonitoringService
{
    private const CACHE_TTL = 3600; // 1 hour

    /**
     * Get overall ad performance metrics
     */
    public function getSystemMetrics(): array
    {
        return Cache::remember('ad_system_metrics', self::CACHE_TTL, function () {
            return [
                'total_active_ads' => Ad::active()->count(),
                'total_impressions_24h' => $this->getImpressions24h(),
                'total_clicks_24h' => $this->getClicks24h(),
                'system_ctr' => $this->calculateSystemCTR(),
                'total_revenue_24h' => $this->getRevenue24h(),
                'ads_by_status' => $this->getAdsByStatus(),
                'performance_alerts' => $this->getPerformanceAlerts()
            ];
        });
    }

    /**
     * Get impressions in last 24 hours
     */
    private function getImpressions24h(): int
    {
        return DB::table('ad_impressions')
            ->where('created_at', '>=', now()->subDay())
            ->count();
    }

    /**
     * Get clicks in last 24 hours
     */
    private function getClicks24h(): int
    {
        return DB::table('ad_clicks')
            ->where('created_at', '>=', now()->subDay())
            ->count();
    }

    /**
     * Calculate system-wide CTR
     */
    private function calculateSystemCTR(): float
    {
        $impressions = $this->getImpressions24h();
        $clicks = $this->getClicks24h();

        return $impressions > 0 ? ($clicks / $impressions) * 100 : 0;
    }

    /**
     * Get revenue in last 24 hours
     */
    private function getRevenue24h(): float
    {
        return DB::table('ad_clicks')
            ->join('ads', 'ad_clicks.ad_id', '=', 'ads.id')
            ->join('ad_plans', 'ads.plan_id', '=', 'ad_plans.id')
            ->where('ad_clicks.created_at', '>=', now()->subDay())
            ->sum('ad_plans.cost_per_click');
    }

    /**
     * Get count of ads by status
     */
    private function getAdsByStatus(): array
    {
        return Ad::select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();
    }

    /**
     * Get performance alerts
     */
    public function getPerformanceAlerts(): array
    {
        $alerts = [];

        // Check for low CTR ads
        $lowCtrAds = Ad::active()
            ->where('impressions', '>', 100)
            ->whereRaw('clicks / impressions < 0.01')
            ->count();
        if ($lowCtrAds > 0) {
            $alerts[] = [
                'type' => 'warning',
                'message' => "{$lowCtrAds} active ads have CTR below 1%"
            ];
        }

        // Check for ads with exhausted budget
        $exhaustedBudgetAds = Ad::active()
            ->whereNotNull('budget')
            ->whereRaw('spent >= budget')
            ->count();
        if ($exhaustedBudgetAds > 0) {
            $alerts[] = [
                'type' => 'error',
                'message' => "{$exhaustedBudgetAds} active ads have exhausted their budget"
            ];
        }

        // Check for ads expiring soon
        $expiringAds = Ad::active()
            ->whereNotNull('end_date')
            ->where('end_date', '<=', now()->addDays(3))
            ->count();
        if ($expiringAds > 0) {
            $alerts[] = [
                'type' => 'info',
                'message' => "{$expiringAds} ads are expiring in the next 3 days"
            ];
        }

        return $alerts;
    }

    /**
     * Log abnormal ad activity
     */
    public function monitorAdActivity(Ad $ad): void
    {
        $stats = app(AdService::class)->getAdStats($ad);
        
        // Check for sudden CTR spikes
        if ($stats['today_ctr'] > 20 && $stats['today_impressions'] > 50) {
            Log::warning('Unusually high CTR detected', [
                'ad_id' => $ad->id ?? $ad->getKey(),
                'today_ctr' => $stats['today_ctr'],
                'today_impressions' => $stats['today_impressions']
            ]);
        }

        // Check for rapid budget depletion
        if (isset($ad->budget) && isset($ad->spent)) {
            $spentToday = $stats['today_clicks'] * ($ad->plan ? $ad->plan->cost_per_click : 0);
            $dailyBudget = $ad->budget / 30; // Assuming monthly budget

            if ($spentToday > $dailyBudget * 2) {
                Log::warning('Rapid budget depletion detected', [
                    'ad_id' => $ad->id ?? $ad->getKey(),
                    'spent_today' => $spentToday,
                    'daily_budget' => $dailyBudget
                ]);
            }
        }
    }

    /**
     * Get health status of the ad system
     */
    public function getSystemHealth(): array
    {
        $metrics = $this->getSystemMetrics();
        $status = 'healthy';
        $issues = [];

        // Check system CTR
        if ($metrics['system_ctr'] < 0.5) {
            $status = 'warning';
            $issues[] = 'System-wide CTR is below 0.5%';
        }

        // Check active ads count
        if ($metrics['total_active_ads'] < 10) {
            $status = 'warning';
            $issues[] = 'Low number of active ads';
        }

        // Check for performance alerts
        if (count($metrics['performance_alerts']) > 5) {
            $status = 'warning';
            $issues[] = 'High number of performance alerts';
        }

        return [
            'status' => $status,
            'issues' => $issues,
            'metrics' => $metrics
        ];
    }
}