<?php

namespace App\Services;

use App\Models\Ad;
use App\Models\User;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class AdService
{
    private const CACHE_TTL = 300; // 5 minutes
    /**
     * Get active ads for a user based on targeting rules
     *
     * @param User|null $user
     * @param string $type
     * @param string|null $position
     * @return Collection
     */
    public function getActiveAds(?User $user = null, string $type = 'banner', ?string $position = null): Collection
    {
        $query = Ad::active()->byType($type);

        if ($position) {
            $query->byPosition($position);
        }

        $ads = $query->get();

        // Filter by targeting rules
        return $ads->filter(function ($ad) use ($user) {
            return $ad->canTargetUser($user);
        });
    }

    /**
     * Record ad impression
     *
     * @param Ad $ad
     * @param string|null $route
     * @return bool
     */
    public function recordImpression(Ad $ad, ?string $route = null): bool
    {
        try {
            // Check if we've shown this ad to this user/IP too recently
            $ipHash = hash('sha256', request()->ip());
            $recentKey = "ad_impression:{$ad->id}:{$ipHash}";
            if (Cache::has($recentKey)) {
                return false; // skip duplicate impressions
            }
            Cache::put($recentKey, true, now()->addMinutes(15));

            DB::table('ad_impressions')->insert([
                'ad_id' => $ad->id,
                'user_id' => auth()->id(),
                'ip_hash' => $ipHash,
                'route' => $route,
                'user_agent' => request()->userAgent(),
                'recorded_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Clear ads cache to respect impression limits
            Cache::tags(['ads'])->flush();
            $ad->recordImpression();
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to record ad impression', [
                'ad_id' => $ad->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Record ad click
     *
     * @param Ad $ad
     * @return bool
     */
    public function recordClick(Ad $ad): bool
    {
        try {
            // Check if we've recorded a click from this user/IP too recently
            $ipHash = hash('sha256', request()->ip());
            $recentKey = "ad_click:{$ad->id}:{$ipHash}";
            if (Cache::has($recentKey)) {
                return false; // skip duplicate clicks
            }
            Cache::put($recentKey, true, now()->addMinutes(30));

            DB::table('ad_clicks')->insert([
                'ad_id' => $ad->id,
                'user_id' => auth()->id(),
                'ip_hash' => $ipHash,
                'target_url' => $ad->target_url,
                'user_agent' => request()->userAgent(),
                'recorded_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $ad->recordClick();
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to record ad click', [
                'ad_id' => $ad->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get ad performance stats with caching
     *
     * @param Ad $ad
     * @return array
     */
    public function getAdStats(Ad $ad): array
    {
        $cacheKey = "ad_metrics:{$ad->id}";
        return Cache::remember($cacheKey, self::CACHE_TTL, function () use ($ad) {
            $today = now()->startOfDay();
            $todayImpressions = DB::table('ad_impressions')
                ->where('ad_id', $ad->id)
                ->where('recorded_at', '>=', $today)
                ->count();

            $todayClicks = DB::table('ad_clicks')
                ->where('ad_id', $ad->id)
                ->where('recorded_at', '>=', $today)
                ->count();

            $todayCtr = $todayImpressions > 0 ? ($todayClicks / $todayImpressions) * 100 : 0;

            return [
                'total_impressions' => $ad->impressions,
                'total_clicks' => $ad->clicks,
                'click_through_rate' => $ad->click_through_rate,
                'budget' => $ad->budget,
                'spent' => $ad->spent,
                'remaining_budget' => $ad->remaining_budget,
                'today_impressions' => $todayImpressions,
                'today_clicks' => $todayClicks,
                'today_ctr' => round($todayCtr, 2),
            ];
        });
    }

    /**
     * Check if ad budget is exhausted
     *
     * @param Ad $ad
     * @return bool
     */
    public function isBudgetExhausted(Ad $ad): bool
    {
        return $ad->budget && $ad->spent >= $ad->budget;
    }

    /**
     * Get ads by performance
     *
     * @param string $metric
     * @param string $order
     * @param int $limit
     * @return Collection
     */
    public function getAdsByPerformance(string $metric = 'clicks', string $order = 'desc', int $limit = 10): Collection
    {
        return Ad::orderBy($metric, $order)->limit($limit)->get();
    }

    /**
     * Get ads expiring soon
     *
     * @param int $days
     * @return Collection
     */
    public function getExpiringAds(int $days = 7): Collection
    {
        return Ad::where('end_date', '<=', now()->addDays($days))
                ->where('end_date', '>', now())
                ->orderBy('end_date')
                ->get();
    }

    /**
     * Get ads with low performance
     *
     * @param float $threshold
     * @return Collection
     */
    public function getLowPerformingAds(float $threshold = 0.5): Collection
    {
        return Ad::where('impressions', '>', 100)
                ->having('click_through_rate', '<', $threshold)
                ->orderBy('click_through_rate')
                ->get();
    }

    /**
     * Pause ads with exhausted budget
     *
     * @return int
     */
    public function pauseExhaustedAds(): int
    {
        return Ad::where('budget', '>', 0)
                ->whereRaw('spent >= budget')
                ->where('status', 'active')
                ->update(['status' => 'inactive']);
    }

    /**
     * Activate scheduled ads
     *
     * @return int
     */
    public function activateScheduledAds(): int
    {
        return Ad::where('status', 'scheduled')
                ->where('start_date', '<=', now())
                ->update(['status' => 'active']);
    }

    /**
     * Deactivate expired ads
     *
     * @return int
     */
    public function deactivateExpiredAds(): int
    {
        return Ad::where('status', 'active')
                ->where('end_date', '<', now())
                ->update(['status' => 'inactive']);
    }

    /**
     * Get ad targeting suggestions
     *
     * @return array
     */
    public function getTargetingSuggestions(): array
    {
        return [
            'roles' => ['admin', 'company_owner', 'job_seeker'],
            'company_sizes' => ['1-10', '11-50', '51-200', '201+'],
            'locations' => ['Douala', 'Yaoundé', 'Bamenda', 'Buea', 'Limbe'],
        ];
    }

    /**
     * Validate ad targeting rules
     *
     * @param array $targeting
     * @return array
     */
    public function validateTargeting(array $targeting): array
    {
        $errors = [];
        $suggestions = $this->getTargetingSuggestions();

        if (isset($targeting['roles'])) {
            $invalidRoles = array_diff($targeting['roles'], $suggestions['roles']);
            if (!empty($invalidRoles)) {
                $errors[] = 'Invalid roles: ' . implode(', ', $invalidRoles);
            }
        }

        if (isset($targeting['company_sizes'])) {
            $invalidSizes = array_diff($targeting['company_sizes'], $suggestions['company_sizes']);
            if (!empty($invalidSizes)) {
                $errors[] = 'Invalid company sizes: ' . implode(', ', $invalidSizes);
            }
        }

        if (isset($targeting['locations'])) {
            $invalidLocations = array_diff($targeting['locations'], $suggestions['locations']);
            if (!empty($invalidLocations)) {
                $errors[] = 'Invalid locations: ' . implode(', ', $invalidLocations);
            }
        }

        return $errors;
    }
}