<?php

namespace App\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

/**
 * Enhanced Mesomb Payment Service
 * 
 * Implements comprehensive Mesomb API integration with proper error handling,
 * security measures, and support for multiple payment methods.
 */
class MesombPaymentService
{
    private const API_BASE_URL = 'https://mesomb.hachther.com/api/';
    private const API_VERSION = 'v1.0';
    
    private Client $httpClient;
    private string $appKey;
    private string $accessKey;
    private string $secretKey;
    private array $config;

    public function __construct()
    {
        $this->config = config('services.mesomb', []);
        $this->appKey = $this->config['app_key'] ?? env('MESOMB_APP_KEY');
        $this->accessKey = $this->config['access_key'] ?? env('MESOMB_ACCESS_KEY');
        $this->secretKey = $this->config['secret_key'] ?? env('MESOMB_SECRET_KEY');
        
        $this->httpClient = new Client([
            'base_uri' => self::API_BASE_URL . self::API_VERSION . '/',
            'timeout' => 60,
            'connect_timeout' => 10,
            'headers' => [
                'User-Agent' => 'HireCameroon/1.0',
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'verify' => env('MESOMB_SSL_VERIFY', true)
        ]);
    }

    /**
     * Initialize a payment collection
     */
    public function initiatePayment(array $paymentData): array
    {
        try {
            // Validate required fields
            $this->validatePaymentData($paymentData);
            
            // Prepare payment payload
            $payload = $this->preparePaymentPayload($paymentData);
            
            // Make API request
            $response = $this->httpClient->post('payment/collect/', [
                'headers' => $this->getAuthHeaders(),
                'json' => $payload,
            ]);

            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($response->getStatusCode() !== 200) {
                throw new \Exception('Payment initiation failed: ' . ($result['message'] ?? 'Unknown error'));
            }

            Log::info('Mesomb payment initiated', [
                'payment_id' => $result['payment_id'] ?? null,
                'reference' => $result['trxref'] ?? null,
                'amount' => $paymentData['amount'] ?? null
            ]);

            return [
                'success' => true,
                'payment_id' => $result['payment_id'] ?? null,
                'reference' => $result['trxref'] ?? Str::uuid()->toString(),
                'payment_url' => $result['payment_url'] ?? null,
                'instructions' => $result['instructions'] ?? [],
                'expires_at' => $this->calculateExpiry(),
            ];

        } catch (GuzzleException $e) {
            Log::error('Mesomb API request failed', [
                'error' => $e->getMessage(),
                'payment_data' => $paymentData
            ]);
            
            return [
                'success' => false,
                'error' => 'Payment service temporarily unavailable',
                'error_code' => 'API_ERROR'
            ];
        } catch (\Exception $e) {
            Log::error('Payment initiation failed', [
                'error' => $e->getMessage(),
                'payment_data' => $paymentData
            ]);
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'error_code' => 'VALIDATION_ERROR'
            ];
        }
    }

    /**
     * Check payment status
     */
    public function checkPaymentStatus(string $reference): array
    {
        try {
            $response = $this->httpClient->get("payment/status/{$reference}", [
                'headers' => $this->getAuthHeaders()
            ]);

            $status = json_decode($response->getBody()->getContents(), true);
            
            Log::info('Payment status checked', [
                'reference' => $reference,
                'status' => $status['status'] ?? null
            ]);

            return [
                'success' => true,
                'status' => $status['status'] ?? 'unknown',
                'amount' => $status['amount'] ?? null,
                'currency' => $status['currency'] ?? 'XAF',
                'created_at' => $status['created'] ?? null,
                'metadata' => $status['metadata'] ?? []
            ];

        } catch (GuzzleException $e) {
            Log::error('Payment status check failed', [
                'reference' => $reference,
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'error' => 'Failed to check payment status'
            ];
        }
    }

    /**
     * Request refund
     */
    public function refundPayment(string $paymentId, ?string $reason = null): array
    {
        try {
            $payload = [
                'payment_id' => $paymentId,
                'reason' => $reason ?? 'Customer request',
            ];

            $response = $this->httpClient->post('payment/refund/', [
                'headers' => $this->getAuthHeaders(),
                'json' => $payload,
            ]);

            $result = json_decode($response->getBody()->getContents(), true);
            
            Log::info('Refund processed', [
                'payment_id' => $paymentId,
                'reason' => $reason,
                'result' => $result
            ]);

            return [
                'success' => true,
                'refund_id' => $result['refund_id'] ?? null,
                'status' => $result['status'] ?? 'processed'
            ];

        } catch (GuzzleException $e) {
            Log::error('Refund request failed', [
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'error' => 'Refund request failed'
            ];
        }
    }

    /**
     * Get supported payment methods
     */
    public function getSupportedPaymentMethods(): array
    {
        return [
            'mobile' => [
                'MTN' => [
                    'name' => 'MTN Mobile Money',
                    'code' => 'MTN',
                    'fee' => 1.5,
                    'currency' => ['XAF'],
                    'min_amount' => 100,
                    'max_amount' => 2000000
                ],
                'MOOV' => [
                    'name' => 'Moov Money',
                    'code' => 'MOOV',
                    'fee' => 1.5,
                    'currency' => ['XAF'],
                    'min_amount' => 100,
                    'max_amount' => 2000000
                ]
            ],
            'cards' => [
                'visa' => [
                    'name' => 'Visa Cards',
                    'code' => 'VISA',
                    'fee' => 2.5,
                    'currency' => ['XAF'],
                    'min_amount' => 500,
                    'max_amount' => 5000000
                ],
                'mastercard' => [
                    'name' => 'Mastercard',
                    'code' => 'MASTERCARD',
                    'fee' => 2.5,
                    'currency' => ['XAF'],
                    'min_amount' => 500,
                    'max_amount' => 5000000
                ]
            ]
        ];
    }

    /**
     * Calculate transaction fee
     */
    public function calculateFee(string $method, float $amount): float
    {
        $methods = $this->getSupportedPaymentMethods();
        
        foreach ($methods as $category => $categoryMethods) {
            if (isset($categoryMethods[$method])) {
                $config = $categoryMethods[$method];
                return ($amount * $config['fee']) / 100;
            }
        }
        
        return 0;
    }

    /**
     * Verify webhook signature
     */
    public function verifyWebhookSignature(string $signature, string $timestamp, string $rawBody): bool
    {
        try {
            $expectedSignature = hash_hmac('sha256', $timestamp . $rawBody, $this->secretKey);
            
            return hash_equals($expectedSignature, $signature);
        } catch (\Exception $e) {
            Log::error('Webhook signature verification failed', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Format amount for API (convert to cents)
     */
    private function formatAmount(float $amount): int
    {
        return (int) round($amount * 100);
    }

    /**
     * Generate unique transaction reference
     */
    private function generateReference(): string
    {
        return 'HIREC_' . strtoupper(Str::random(12));
    }

    /**
     * Calculate payment expiry (default 30 minutes)
     */
    private function calculateExpiry(int $minutes = 30): string
    {
        return now()->addMinutes($minutes)->toISOString();
    }

    /**
     * Get authentication headers
     */
    private function getAuthHeaders(): array
    {
        return [
            'X-MeSomb-Application' => $this->appKey,
            'X-MeSomb-Access-Key' => $this->accessKey,
            'X-MeSomb-Secret-Key' => $this->secretKey,
        ];
    }

    /**
     * Validate payment data
     */
    private function validatePaymentData(array $data): void
    {
        $required = ['amount', 'currency', 'phone'];
        
        foreach ($required as $field) {
            if (empty($data[$field])) {
                throw new \InvalidArgumentException("Missing required field: {$field}");
            }
        }
        
        if (!is_numeric($data['amount']) || $data['amount'] <= 0) {
            throw new \InvalidArgumentException('Invalid amount');
        }
        
        if (!in_array($data['currency'], ['XAF', 'USD', 'EUR'])) {
            throw new \InvalidArgumentException('Unsupported currency');
        }
    }

    /**
     * Prepare payment payload for API
     */
    private function preparePaymentPayload(array $data): array
    {
        return [
            'amount' => $this->formatAmount($data['amount']),
            'currency' => $data['currency'],
            'phone' => $data['phone'],
            'trxref' => $data['reference'] ?? $this->generateReference(),
            'reason' => $data['reason'] ?? 'Payment for services',
            'email' => $data['email'] ?? null,
            'name' => $data['name'] ?? null,
            'success_url' => $data['success_url'] ?? config('app.url') . '/payment/success',
            'fail_url' => $data['fail_url'] ?? config('app.url') . '/payment/failed',
            'callback_url' => $data['callback_url'] ?? config('app.url') . '/webhooks/mesomb',
            'environment' => config('app.env') === 'production' ? 'live' : 'test',
        ];
    }
}