<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\User;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Send a notification to a user
     *
     * @param User $user
     * @param string $type
     * @param string $title
     * @param string $message
     * @param array $data
     * @param User|null $sender
     * @param mixed|null $related
     * @param string $priority
     * @return Notification
     */
    public function send(
        User $user,
        string $type,
        string $title,
        string $message,
        array $data = [],
        ?User $sender = null,
        $related = null,
        string $priority = 'normal'
    ): Notification {
        $notification = Notification::create([
            'type' => $type,
            'title' => $title,
            'message' => $message,
            'data' => $data,
            'user_id' => $user->id,
            'sender_id' => $sender?->id,
            'related_type' => $related ? get_class($related) : null,
            'related_id' => $related?->id ?? null,
            'priority' => $priority,
        ]);

        // Send email if type is email
        if ($type === 'email') {
            $this->sendEmailNotification($notification);
        }

        return $notification;
    }

    /**
     * Send bulk notifications
     *
     * @param array $users
     * @param string $type
     * @param string $title
     * @param string $message
     * @param array $data
     * @param User|null $sender
     * @param mixed|null $related
     * @param string $priority
     * @return array
     */
    public function sendBulk(
        array $users,
        string $type,
        string $title,
        string $message,
        array $data = [],
        ?User $sender = null,
        $related = null,
        string $priority = 'normal'
    ): array {
        $notifications = [];

        foreach ($users as $user) {
            $notifications[] = $this->send($user, $type, $title, $message, $data, $sender, $related, $priority);
        }

        return $notifications;
    }

    /**
     * Schedule a notification for later
     *
     * @param User $user
     * @param string $type
     * @param string $title
     * @param string $message
     * @param \Carbon\Carbon $scheduledAt
     * @param array $data
     * @param User|null $sender
     * @param mixed|null $related
     * @param string $priority
     * @return Notification
     */
    public function schedule(
        User $user,
        string $type,
        string $title,
        string $message,
        $scheduledAt,
        array $data = [],
        ?User $sender = null,
        $related = null,
        string $priority = 'normal'
    ): Notification {
        return Notification::create([
            'type' => $type,
            'title' => $title,
            'message' => $message,
            'data' => $data,
            'user_id' => $user->id,
            'sender_id' => $sender?->id,
            'related_type' => $related ? get_class($related) : null,
            'related_id' => $related?->id ?? null,
            'priority' => $priority,
            'status' => 'scheduled',
            'scheduled_at' => $scheduledAt,
        ]);
    }

    /**
     * Process scheduled notifications
     *
     * @return void
     */
    public function processScheduledNotifications(): void
    {
        $scheduledNotifications = Notification::scheduled()
            ->where('scheduled_at', '<=', now())
            ->get();

        foreach ($scheduledNotifications as $notification) {
            $notification->update(['status' => 'unread', 'scheduled_at' => null]);

            if ($notification->type === 'email') {
                $this->sendEmailNotification($notification);
            }
        }
    }

    /**
     * Send email notification
     *
     * @param Notification $notification
     * @return void
     */
    private function sendEmailNotification(Notification $notification): void
    {
        try {
            Mail::raw($notification->message, function ($message) use ($notification) {
                $message->to($notification->user->email)
                        ->subject($notification->title)
                        ->from(config('mail.from.address'), config('mail.from.name'));
            });
        } catch (\Exception $e) {
            Log::error('Failed to send email notification: ' . $e->getMessage(), [
                'notification_id' => $notification->id,
                'user_id' => $notification->user_id,
            ]);
        }
    }

    /**
     * Mark notification as read
     *
     * @param Notification $notification
     * @return void
     */
    public function markAsRead(Notification $notification): void
    {
        $notification->markAsRead();
    }

    /**
     * Mark all user notifications as read
     *
     * @param User $user
     * @return void
     */
    public function markAllAsRead(User $user): void
    {
        $user->notifications()->unread()->update([
            'status' => 'read',
            'read_at' => now(),
        ]);
    }

    /**
     * Get user notifications
     *
     * @param User $user
     * @param string $status
     * @param int $limit
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getUserNotifications(User $user, string $status = 'all', int $limit = 50)
    {
        $query = $user->notifications()->orderBy('created_at', 'desc');

        if ($status !== 'all') {
            $query->where('status', $status);
        }

        return $query->limit($limit)->get();
    }

    /**
     * Get unread count for user
     *
     * @param User $user
     * @return int
     */
    public function getUnreadCount(User $user): int
    {
        return $user->notifications()->unread()->count();
    }

    /**
     * Delete old notifications
     *
     * @param int $daysOld
     * @return int
     */
    public function cleanupOldNotifications(int $daysOld = 90): int
    {
        return Notification::where('created_at', '<', now()->subDays($daysOld))
                          ->where('status', 'read')
                          ->delete();
    }
}