<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     * This migration creates the optimized database structure with 12 core tables
     * replacing the previous 35+ table structure.
     */
    public function up()
    {
        // Note: Tables are already created by database_cleanup.php script
        // This migration serves as documentation and ensures the structure exists

        // 1. Users Table (Enhanced)
        if (!Schema::hasTable('users_optimized')) {
            Schema::create('users_optimized', function (Blueprint $table) {
                $table->id();
                $table->string('name');
                $table->string('email')->unique();
                $table->timestamp('email_verified_at')->nullable();
                $table->string('password');
                $table->rememberToken();
                $table->enum('account_type', ['job_seeker', 'company_owner', 'admin'])->default('job_seeker');
                $table->enum('role', ['user', 'admin', 'moderator'])->default('user');
                $table->string('status')->default('active');
                $table->string('phone', 20)->nullable();
                $table->string('location')->nullable();
                $table->text('bio')->nullable();
                $table->string('avatar')->nullable();
                $table->date('date_of_birth')->nullable();
                $table->enum('gender', ['male', 'female', 'other', 'prefer_not_to_say'])->nullable();
                $table->string('current_position')->nullable();
                $table->integer('years_experience')->nullable();
                $table->string('industry')->nullable();
                $table->string('highest_education_level')->nullable();
                $table->json('education_details')->nullable();
                $table->json('profile_data')->nullable();
                $table->json('email_notifications')->nullable();
                $table->json('privacy_settings')->nullable();
                $table->json('preferences')->nullable();
                $table->timestamps();
                $table->index(['account_type', 'status']);
                $table->index(['location', 'industry']);
                $table->index('email');
            });
        }

        // 2. Organizations Table
        if (!Schema::hasTable('organizations_optimized')) {
            Schema::create('organizations_optimized', function (Blueprint $table) {
                $table->id();
                $table->string('name');
                $table->string('slug')->unique();
                $table->text('description')->nullable();
                $table->string('sector')->nullable();
                $table->string('contact_email')->nullable();
                $table->string('contact_phone', 20)->nullable();
                $table->text('address')->nullable();
                $table->string('website')->nullable();
                $table->string('logo')->nullable();
                $table->string('banner_image')->nullable();
                $table->string('registration_number')->nullable();
                $table->string('tax_id')->nullable();
                $table->unsignedBigInteger('owner_user_id')->nullable();
                $table->enum('status', ['active', 'inactive', 'pending', 'suspended'])->default('active');
                $table->string('subscription_plan')->nullable();
                $table->timestamp('subscription_expires_at')->nullable();
                $table->json('branding')->nullable();
                $table->json('settings')->nullable();
                $table->json('domains')->nullable();
                $table->json('custom_features')->nullable();
                $table->json('seo_data')->nullable();
                $table->json('marketing_settings')->nullable();
                $table->timestamps();
                $table->index(['sector', 'status']);
                $table->index(['subscription_plan', 'status']);
                $table->index('owner_user_id');
                $table->foreign('owner_user_id')->references('id')->on('users_optimized')->onDelete('set null');
            });
        }

        // 3. Jobs Table (Enhanced)
        if (!Schema::hasTable('jobs_optimized')) {
            Schema::create('jobs_optimized', function (Blueprint $table) {
                $table->id();
                $table->foreignId('organization_id')->constrained('organizations_optimized')->onDelete('cascade');
                $table->string('title');
                $table->string('slug')->unique();
                $table->longText('description');
                $table->string('location_region')->nullable();
                $table->string('location_city')->nullable();
                $table->enum('employment_type', ['full_time', 'part_time', 'contract', 'freelance', 'internship'])->default('full_time');
                $table->enum('experience_level', ['entry', 'mid', 'senior', 'executive'])->default('entry');
                $table->unsignedInteger('salary_min')->nullable();
                $table->unsignedInteger('salary_max')->nullable();
                $table->string('currency', 3)->default('XAF');
                $table->boolean('negotiable')->default(false);
                $table->timestamp('application_deadline')->nullable();
                $table->timestamp('featured_until')->nullable();
                $table->text('requirements')->nullable();
                $table->json('benefits')->nullable();
                $table->string('application_url')->nullable();
                $table->boolean('verified_company')->default(false);
                $table->json('scam_flags')->nullable();
                $table->enum('verification_status', ['pending', 'verified', 'rejected'])->default('pending');
                $table->enum('status', ['draft', 'active', 'paused', 'closed', 'expired'])->default('active');
                $table->json('seo_data')->nullable();
                $table->timestamps();
                $table->index(['organization_id', 'status']);
                $table->index(['location_city', 'employment_type']);
                $table->index(['featured_until', 'application_deadline']);
            });
        }

        // 4. Applications Table (Enhanced)
        if (!Schema::hasTable('applications_optimized')) {
            Schema::create('applications_optimized', function (Blueprint $table) {
                $table->id();
                $table->foreignId('job_id')->constrained('jobs_optimized')->onDelete('cascade');
                $table->foreignId('user_id')->constrained('users_optimized')->onDelete('cascade');
                $table->enum('status', ['pending', 'reviewed', 'interview', 'hired', 'rejected'])->default('pending');
                $table->text('cover_letter')->nullable();
                $table->string('resume_path')->nullable();
                $table->string('portfolio_url')->nullable();
                $table->integer('salary_expectation')->nullable();
                $table->date('available_start_date')->nullable();
                $table->json('application_data')->nullable();
                $table->timestamp('interview_scheduled_at')->nullable();
                $table->text('notes')->nullable();
                $table->tinyInteger('rating')->nullable();
                $table->timestamps();
                $table->index(['job_id', 'status']);
                $table->index(['user_id', 'status']);
                $table->index(['status', 'created_at']);
            });
        }

        // 5. Content Table (Universal)
        if (!Schema::hasTable('content_optimized')) {
            Schema::create('content_optimized', function (Blueprint $table) {
                $table->id();
                $table->enum('type', ['ad', 'news', 'announcement', 'email_template', 'faq', 'featured'])->default('ad');
                $table->string('title');
                $table->string('slug')->nullable()->unique();
                $table->longText('content')->nullable();
                $table->text('excerpt')->nullable();
                $table->string('image_path')->nullable();
                $table->string('target_url')->nullable();
                $table->string('position')->nullable();
                $table->foreignId('user_id')->nullable()->constrained('users_optimized')->nullOnDelete();
                $table->foreignId('organization_id')->nullable()->constrained('organizations_optimized')->nullOnDelete();
                $table->foreignId('related_job_id')->nullable()->constrained('jobs_optimized')->nullOnDelete();
                $table->enum('status', ['draft', 'published', 'archived'])->default('draft');
                $table->json('seo_data')->nullable();
                $table->json('metadata')->nullable();
                $table->integer('display_order')->default(0);
                $table->boolean('is_featured')->default(false);
                $table->timestamp('expires_at')->nullable();
                $table->timestamps();
                $table->index(['type', 'status']);
                $table->index(['is_featured', 'expires_at']);
                $table->index(['user_id', 'type']);
            });
        }

        // 6. Interactions Table (Social Features)
        if (!Schema::hasTable('interactions_optimized')) {
            Schema::create('interactions_optimized', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->constrained('users_optimized')->onDelete('cascade');
                $table->string('target_type');
                $table->unsignedBigInteger('target_id');
                $table->enum('interaction_type', [
                    'like', 'unlike', 'comment', 'reply', 'share', 'view',
                    'report', 'review', 'rating', 'follow', 'bookmark'
                ]);
                $table->text('content')->nullable();
                $table->tinyInteger('rating')->nullable();
                $table->foreignId('parent_id')->nullable()->constrained('interactions_optimized')->nullOnDelete();
                $table->enum('status', ['active', 'hidden', 'reported', 'approved', 'rejected'])->default('active');
                $table->text('moderation_notes')->nullable();
                $table->json('metadata')->nullable();
                $table->json('interaction_data')->nullable();
                $table->timestamps();
                $table->index(['user_id', 'interaction_type'], 'idx_int_opt_user_it');
                $table->index(['target_type', 'target_id', 'interaction_type'], 'idx_int_opt_tt_tid_it');
                $table->index(['status', 'created_at'], 'idx_int_opt_status_created');
                $table->index('parent_id', 'idx_int_opt_parent');
                $table->index(['target_type', 'target_id', 'user_id', 'interaction_type'], 'idx_int_opt_tt_tid_uid_it');
            });
        }

        // 7. Transactions Table (Unified Billing)
        if (!Schema::hasTable('transactions_optimized')) {
            Schema::create('transactions_optimized', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->nullable()->constrained('users_optimized')->nullOnDelete();
                $table->foreignId('organization_id')->nullable()->constrained('organizations_optimized')->nullOnDelete();
                $table->enum('type', [
                    'subscription_payment', 'job_payment', 'refund', 'api_usage',
                    'premium_feature', 'advertisement', 'bulk_email', 'custom'
                ]);
                $table->decimal('amount', 10, 2)->nullable();
                $table->string('currency', 3)->default('XAF');
                $table->enum('status', ['pending', 'completed', 'failed', 'cancelled', 'refunded'])->default('pending');
                $table->string('provider')->nullable();
                $table->string('provider_transaction_id')->nullable();
                $table->json('provider_data')->nullable();
                $table->string('subscription_plan')->nullable();
                $table->dateTime('billing_period_start')->nullable();
                $table->dateTime('billing_period_end')->nullable();
                $table->json('plan_features')->nullable();
                $table->string('api_key_id')->nullable();
                $table->integer('api_usage_quota')->nullable();
                $table->integer('api_usage_current')->default(0);
                $table->dateTime('api_usage_reset_date')->nullable();
                $table->json('api_usage_details')->nullable();
                $table->json('metadata')->nullable();
                $table->text('description')->nullable();
                $table->text('notes')->nullable();
                $table->timestamps();
                $table->index(['user_id', 'status']);
                $table->index(['organization_id', 'type']);
                $table->index(['status', 'created_at']);
                $table->index(['provider', 'provider_transaction_id']);
                $table->index('subscription_plan');
            });
        }

        // 8. Skills Table (Flexible)
        if (!Schema::hasTable('skills_optimized')) {
            Schema::create('skills_optimized', function (Blueprint $table) {
                $table->id();
                $table->string('name');
                $table->string('category')->nullable();
                $table->text('description')->nullable();
                $table->boolean('is_verified')->default(false);
                $table->foreignId('user_id')->nullable()->constrained('users_optimized')->nullOnDelete();
                $table->integer('popularity_score')->default(0);
                $table->json('related_skills')->nullable();
                $table->timestamps();
                $table->index(['category', 'is_verified']);
                $table->index('popularity_score');
            });
        }

        // 9. Saved Items Table
        if (!Schema::hasTable('saved_items_optimized')) {
            Schema::create('saved_items_optimized', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->constrained('users_optimized')->onDelete('cascade');
                $table->enum('item_type', ['job', 'company', 'search', 'filter']);
                $table->unsignedBigInteger('item_id')->nullable();
                $table->json('query_data')->nullable();
                $table->string('name')->nullable();
                $table->text('description')->nullable();
                $table->boolean('is_auto_refresh')->default(false);
                $table->timestamp('last_checked_at')->nullable();
                $table->string('notification_frequency')->default('daily');
                $table->timestamps();
                $table->index(['user_id', 'item_type']);
                $table->index(['is_auto_refresh', 'last_checked_at']);
            });
        }

        // 10. Activity Logs Table
        if (!Schema::hasTable('activity_logs_optimized')) {
            Schema::create('activity_logs_optimized', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->nullable()->constrained('users_optimized')->nullOnDelete();
                $table->foreignId('organization_id')->nullable()->constrained('organizations_optimized')->nullOnDelete();
                $table->string('action', 100);
                $table->string('resource_type', 50)->nullable();
                $table->unsignedBigInteger('resource_id')->nullable();
                $table->text('description')->nullable();
                $table->json('old_values')->nullable();
                $table->json('new_values')->nullable();
                $table->string('ip_address', 45)->nullable();
                $table->text('user_agent')->nullable();
                $table->string('session_id', 255)->nullable();
                $table->enum('severity', ['info', 'warning', 'error', 'critical'])->default('info');
                $table->enum('category', ['auth', 'activity', 'admin', 'system', 'security'])->default('activity');
                $table->json('metadata')->nullable();
                $table->timestamps();
                $table->index(['user_id', 'created_at']);
                $table->index(['resource_type', 'resource_id']);
                $table->index(['severity', 'created_at']);
                $table->index('category');
            });
        }

        // 11. Settings Table
        if (!Schema::hasTable('settings_optimized')) {
            Schema::create('settings_optimized', function (Blueprint $table) {
                $table->id();
                $table->string('key')->unique();
                $table->text('value')->nullable();
                $table->enum('type', ['string', 'integer', 'boolean', 'json'])->default('string');
                $table->string('category', 50)->default('general');
                $table->boolean('is_public')->default(false);
                $table->text('description')->nullable();
                $table->json('validation_rules')->nullable();
                $table->foreignId('updated_by_user_id')->nullable()->constrained('users_optimized')->nullOnDelete();
                $table->timestamps();
                $table->index('category');
                $table->index('is_public');
            });
        }

        // 12. Sessions Table (Enhanced)
        if (!Schema::hasTable('sessions_optimized')) {
            Schema::create('sessions_optimized', function (Blueprint $table) {
                $table->string('id')->primary();
                $table->foreignId('user_id')->nullable()->constrained('users_optimized')->onDelete('cascade');
                $table->string('ip_address', 45)->nullable();
                $table->text('user_agent')->nullable();
                $table->longText('payload');
                $table->integer('last_activity');
                $table->boolean('is_active')->default(true);
                $table->string('device_type')->nullable();
                $table->string('location')->nullable();
                $table->string('remember_token')->nullable();
                $table->string('login_method')->default('email');
                $table->timestamps();
                $table->index(['user_id', 'is_active']);
                $table->index('last_activity');
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down()
    {
        // Drop tables in reverse order to handle foreign key constraints
        $tables = [
            'sessions_optimized',
            'settings_optimized',
            'activity_logs_optimized',
            'saved_items_optimized',
            'skills_optimized',
            'transactions_optimized',
            'interactions_optimized',
            'content_optimized',
            'applications_optimized',
            'jobs_optimized',
            'organizations_optimized',
            'users_optimized'
        ];

        foreach ($tables as $table) {
            Schema::dropIfExists($table);
        }
    }
};
