<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     * Enhance existing core tables with JSON fields and better structure
     * instead of creating completely new tables.
     */
    public function up()
    {
        // Enhance users_optimized table with additional fields
        Schema::table('users_optimized', function (Blueprint $table) {
            // Profile enhancement fields
            if (!Schema::hasColumn('users', 'account_type')) {
                $table->enum('account_type', ['job_seeker', 'company_owner', 'admin'])->default('job_seeker')->after('email');
            }
            if (!Schema::hasColumn('users', 'role')) {
                $table->enum('role', ['user', 'admin', 'moderator'])->default('user')->after('account_type');
            }
            if (!Schema::hasColumn('users', 'status')) {
                $table->string('status')->default('active')->after('role');
            }
            if (!Schema::hasColumn('users', 'phone')) {
                $table->string('phone', 20)->nullable()->after('status');
            }
            if (!Schema::hasColumn('users', 'location')) {
                $table->string('location')->nullable()->after('phone');
            }
            if (!Schema::hasColumn('users', 'bio')) {
                $table->text('bio')->nullable()->after('location');
            }
            if (!Schema::hasColumn('users', 'avatar')) {
                $table->string('avatar')->nullable()->after('bio');
            }
            if (!Schema::hasColumn('users', 'date_of_birth')) {
                $table->date('date_of_birth')->nullable()->after('avatar');
            }
            if (!Schema::hasColumn('users', 'gender')) {
                $table->enum('gender', ['male', 'female', 'other', 'prefer_not_to_say'])->nullable()->after('date_of_birth');
            }
            if (!Schema::hasColumn('users', 'current_position')) {
                $table->string('current_position')->nullable()->after('gender');
            }
            if (!Schema::hasColumn('users', 'years_experience')) {
                $table->integer('years_experience')->nullable()->after('current_position');
            }
            if (!Schema::hasColumn('users', 'industry')) {
                $table->string('industry')->nullable()->after('years_experience');
            }
            if (!Schema::hasColumn('users', 'highest_education_level')) {
                $table->string('highest_education_level')->nullable()->after('industry');
            }

            // JSON fields for flexible data
            if (!Schema::hasColumn('users', 'education_details')) {
                $table->json('education_details')->nullable()->after('highest_education_level');
            }
            if (!Schema::hasColumn('users', 'profile_data')) {
                $table->json('profile_data')->nullable()->after('education_details');
            }
            if (!Schema::hasColumn('users', 'email_notifications')) {
                $table->json('email_notifications')->nullable()->after('profile_data');
            }
            if (!Schema::hasColumn('users', 'privacy_settings')) {
                $table->json('privacy_settings')->nullable()->after('email_notifications');
            }
            if (!Schema::hasColumn('users', 'preferences')) {
                $table->json('preferences')->nullable()->after('privacy_settings');
            }

            // Indexes for performance
            $table->index(['account_type', 'status']);
            $table->index(['location', 'industry']);
        });

        // Enhance organizations_optimized table with additional fields
        Schema::table('organizations_optimized', function (Blueprint $table) {
            // Additional company fields
            if (!Schema::hasColumn('companies', 'slug')) {
                $table->string('slug')->unique()->nullable()->after('name');
            }
            if (!Schema::hasColumn('companies', 'contact_email')) {
                $table->string('contact_email')->nullable()->after('slug');
            }
            if (!Schema::hasColumn('companies', 'contact_phone')) {
                $table->string('contact_phone', 20)->nullable()->after('contact_email');
            }
            if (!Schema::hasColumn('companies', 'address')) {
                $table->text('address')->nullable()->after('contact_phone');
            }
            if (!Schema::hasColumn('companies', 'website')) {
                $table->string('website')->nullable()->after('address');
            }
            if (!Schema::hasColumn('companies', 'logo')) {
                $table->string('logo')->nullable()->after('website');
            }
            if (!Schema::hasColumn('companies', 'banner_image')) {
                $table->string('banner_image')->nullable()->after('logo');
            }
            if (!Schema::hasColumn('companies', 'registration_number')) {
                $table->string('registration_number')->nullable()->after('banner_image');
            }
            if (!Schema::hasColumn('companies', 'tax_id')) {
                $table->string('tax_id')->nullable()->after('registration_number');
            }
            if (!Schema::hasColumn('companies', 'status')) {
                $table->enum('status', ['active', 'inactive', 'pending', 'suspended'])->default('active')->after('tax_id');
            }
            if (!Schema::hasColumn('companies', 'subscription_plan')) {
                $table->string('subscription_plan')->nullable()->after('status');
            }
            if (!Schema::hasColumn('companies', 'subscription_expires_at')) {
                $table->timestamp('subscription_expires_at')->nullable()->after('subscription_plan');
            }

            // JSON fields for flexible data
            if (!Schema::hasColumn('companies', 'branding')) {
                $table->json('branding')->nullable()->after('subscription_expires_at');
            }
            if (!Schema::hasColumn('companies', 'settings')) {
                $table->json('settings')->nullable()->after('branding');
            }
            if (!Schema::hasColumn('companies', 'domains')) {
                $table->json('domains')->nullable()->after('settings');
            }
            if (!Schema::hasColumn('companies', 'custom_features')) {
                $table->json('custom_features')->nullable()->after('domains');
            }
            if (!Schema::hasColumn('companies', 'seo_data')) {
                $table->json('seo_data')->nullable()->after('custom_features');
            }
            if (!Schema::hasColumn('companies', 'marketing_settings')) {
                $table->json('marketing_settings')->nullable()->after('seo_data');
            }

            // Indexes for performance
            $table->index(['sector', 'status']);
            $table->index(['subscription_plan', 'status']);
            $table->index('owner_user_id');
        });

        // Enhance jobs_optimized table with additional fields
        Schema::table('jobs_optimized', function (Blueprint $table) {
            // Additional job fields
            if (!Schema::hasColumn('jobs', 'slug')) {
                $table->string('slug')->unique()->nullable()->after('title');
            }
            if (!Schema::hasColumn('jobs', 'employment_type')) {
                $table->enum('employment_type', ['full_time', 'part_time', 'contract', 'freelance', 'internship'])->default('full_time')->after('slug');
            }
            if (!Schema::hasColumn('jobs', 'experience_level')) {
                $table->enum('experience_level', ['entry', 'mid', 'senior', 'executive'])->default('entry')->after('employment_type');
            }
            if (!Schema::hasColumn('jobs', 'currency')) {
                $table->string('currency', 3)->default('XAF')->after('salary_max');
            }
            if (!Schema::hasColumn('jobs', 'negotiable')) {
                $table->boolean('negotiable')->default(false)->after('currency');
            }
            if (!Schema::hasColumn('jobs', 'featured_until')) {
                $table->timestamp('featured_until')->nullable()->after('negotiable');
            }
            if (!Schema::hasColumn('jobs', 'application_url')) {
                $table->string('application_url')->nullable()->after('featured_until');
            }
            if (!Schema::hasColumn('jobs', 'verified_company')) {
                $table->boolean('verified_company')->default(false)->after('application_url');
            }
            if (!Schema::hasColumn('jobs', 'scam_flags')) {
                $table->json('scam_flags')->nullable()->after('verified_company');
            }
            if (!Schema::hasColumn('jobs', 'verification_status')) {
                $table->enum('verification_status', ['pending', 'verified', 'rejected'])->default('pending')->after('scam_flags');
            }

            // JSON fields for flexible data
            if (!Schema::hasColumn('jobs', 'benefits')) {
                $table->json('benefits')->nullable()->after('verification_status');
            }
            if (!Schema::hasColumn('jobs', 'seo_data')) {
                $table->json('seo_data')->nullable()->after('benefits');
            }

            // Indexes for performance
            $table->index(['company_id', 'status']);
            $table->index(['location_city', 'employment_type']);
            $table->index(['featured_until', 'application_deadline']);
        });

        // Enhance applications_optimized table with additional fields
        Schema::table('applications_optimized', function (Blueprint $table) {
            // Additional application fields
            if (!Schema::hasColumn('applications', 'cover_letter')) {
                $table->text('cover_letter')->nullable()->after('status');
            }
            if (!Schema::hasColumn('applications', 'resume_path')) {
                $table->string('resume_path')->nullable()->after('cover_letter');
            }
            if (!Schema::hasColumn('applications', 'portfolio_url')) {
                $table->string('portfolio_url')->nullable()->after('resume_path');
            }
            if (!Schema::hasColumn('applications', 'salary_expectation')) {
                $table->integer('salary_expectation')->nullable()->after('portfolio_url');
            }
            if (!Schema::hasColumn('applications', 'available_start_date')) {
                $table->date('available_start_date')->nullable()->after('salary_expectation');
            }
            if (!Schema::hasColumn('applications', 'interview_scheduled_at')) {
                $table->timestamp('interview_scheduled_at')->nullable()->after('available_start_date');
            }
            if (!Schema::hasColumn('applications', 'rating')) {
                $table->tinyInteger('rating')->nullable()->after('interview_scheduled_at');
            }

            // JSON fields for flexible data
            if (!Schema::hasColumn('applications', 'application_data')) {
                $table->json('application_data')->nullable()->after('rating');
            }

            // Indexes for performance
            $table->index(['job_id', 'status']);
            $table->index(['user_id', 'status']);
            $table->index(['status', 'created_at']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down()
    {
        // Remove added columns from users table
        Schema::table('users', function (Blueprint $table) {
            $table->dropIndex(['account_type', 'status']);
            $table->dropIndex(['location', 'industry']);
            $table->dropColumn([
                'account_type', 'role', 'status', 'phone', 'location', 'bio', 'avatar',
                'date_of_birth', 'gender', 'current_position', 'years_experience', 'industry',
                'highest_education_level', 'education_details', 'profile_data',
                'email_notifications', 'privacy_settings', 'preferences'
            ]);
        });

        // Remove added columns from companies table
        Schema::table('companies', function (Blueprint $table) {
            $table->dropIndex(['sector', 'status']);
            $table->dropIndex(['subscription_plan', 'status']);
            $table->dropIndex(['owner_user_id']);
            $table->dropColumn([
                'slug', 'contact_email', 'contact_phone', 'address', 'website', 'logo',
                'banner_image', 'registration_number', 'tax_id', 'status', 'subscription_plan',
                'subscription_expires_at', 'branding', 'settings', 'domains', 'custom_features',
                'seo_data', 'marketing_settings'
            ]);
        });

        // Remove added columns from jobs table
        Schema::table('jobs', function (Blueprint $table) {
            $table->dropIndex(['company_id', 'status']);
            $table->dropIndex(['location_city', 'employment_type']);
            $table->dropIndex(['featured_until', 'application_deadline']);
            $table->dropColumn([
                'slug', 'employment_type', 'experience_level', 'currency', 'negotiable',
                'featured_until', 'application_url', 'verified_company', 'scam_flags',
                'verification_status', 'benefits', 'seo_data'
            ]);
        });

        // Remove added columns from applications table
        Schema::table('applications', function (Blueprint $table) {
            $table->dropIndex(['job_id', 'status']);
            $table->dropIndex(['user_id', 'status']);
            $table->dropIndex(['status', 'created_at']);
            $table->dropColumn([
                'cover_letter', 'resume_path', 'portfolio_url', 'salary_expectation',
                'available_start_date', 'interview_scheduled_at', 'rating', 'application_data'
            ]);
        });
    }
};
