// HireCameroon Documentation JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Navigation functionality
    initNavigation();
    initScrollSpy();
    initCodeTabs();
    initSmoothScrolling();

    // Initialize animations
    initScrollAnimations();
});

// Navigation Toggle for Mobile
function initNavigation() {
    const navToggle = document.querySelector('.nav-toggle');
    const navMenu = document.querySelector('.nav-menu');

    if (navToggle && navMenu) {
        navToggle.addEventListener('click', function() {
            navMenu.classList.toggle('active');
            navToggle.classList.toggle('active');
        });

        // Close menu when clicking outside
        document.addEventListener('click', function(e) {
            if (!navToggle.contains(e.target) && !navMenu.contains(e.target)) {
                navMenu.classList.remove('active');
                navToggle.classList.remove('active');
            }
        });
    }
}

// Smooth Scrolling for Navigation Links
function initSmoothScrolling() {
    const navLinks = document.querySelectorAll('.nav-link');

    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();

            const targetId = this.getAttribute('href');
            const targetSection = document.querySelector(targetId);

            if (targetSection) {
                const offsetTop = targetSection.offsetTop - 70; // Account for fixed nav

                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });

                // Update active nav link
                navLinks.forEach(link => link.classList.remove('active'));
                this.classList.add('active');

                // Close mobile menu
                document.querySelector('.nav-menu').classList.remove('active');
                document.querySelector('.nav-toggle').classList.remove('active');
            }
        });
    });
}

// Scroll Spy for Navigation
function initScrollSpy() {
    const sections = document.querySelectorAll('.section');
    const navLinks = document.querySelectorAll('.nav-link');

    window.addEventListener('scroll', function() {
        let current = '';

        sections.forEach(section => {
            const sectionTop = section.offsetTop - 100;
            const sectionHeight = section.clientHeight;

            if (pageYOffset >= sectionTop) {
                current = section.getAttribute('id');
            }
        });

        navLinks.forEach(link => {
            link.classList.remove('active');
            if (link.getAttribute('href') === '#' + current) {
                link.classList.add('active');
            }
        });
    });
}

// Code Tabs Functionality
function initCodeTabs() {
    const codeTabs = document.querySelectorAll('.code-tab');
    const codeBlocks = document.querySelectorAll('.code-block');

    codeTabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const tabId = this.getAttribute('data-tab');

            // Remove active class from all tabs
            codeTabs.forEach(t => t.classList.remove('active'));
            // Add active class to clicked tab
            this.classList.add('active');

            // Hide all code blocks
            codeBlocks.forEach(block => block.classList.remove('active'));
            // Show selected code block
            document.getElementById(tabId + '-code').classList.add('active');
        });
    });
}

// Scroll Animations
function initScrollAnimations() {
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-in');
            }
        });
    }, observerOptions);

    // Observe elements for animation
    const animateElements = document.querySelectorAll('.feature-card, .table-card, .guide-card, .workflow-step, .status-card, .endpoint-group, .tech-item, .arch-layer, .stat-card, .feature-item');

    animateElements.forEach(element => {
        observer.observe(element);
    });
}

// Add CSS for animations
const style = document.createElement('style');
style.textContent = `
    .feature-card, .table-card, .guide-card, .workflow-step, .status-card, .endpoint-group, .tech-item, .arch-layer, .stat-card, .feature-item {
        opacity: 0;
        transform: translateY(20px);
        transition: opacity 0.6s ease, transform 0.6s ease;
    }

    .feature-card.animate-in, .table-card.animate-in, .guide-card.animate-in, .workflow-step.animate-in, .status-card.animate-in, .endpoint-group.animate-in, .tech-item.animate-in, .arch-layer.animate-in, .stat-card.animate-in, .feature-item.animate-in {
        opacity: 1;
        transform: translateY(0);
    }

    .nav-menu.active {
        display: flex;
        flex-direction: column;
        position: absolute;
        top: 70px;
        left: 0;
        right: 0;
        background: white;
        border-bottom: 1px solid var(--border-color);
        padding: 20px;
        box-shadow: var(--shadow);
    }

    .nav-toggle.active {
        transform: rotate(90deg);
    }

    @media (min-width: 769px) {
        .nav-menu.active {
            display: flex;
            flex-direction: row;
            position: static;
            background: transparent;
            border: none;
            padding: 0;
            box-shadow: none;
        }
    }
`;
document.head.appendChild(style);

// Typing Animation for Hero Code
function initTypingAnimation() {
    const codeElement = document.querySelector('.code-content code');
    if (!codeElement) return;

    const text = codeElement.textContent;
    codeElement.textContent = '';

    let i = 0;
    const timer = setInterval(function() {
        if (i < text.length) {
            codeElement.textContent += text.charAt(i);
            i++;
        } else {
            clearInterval(timer);
        }
    }, 50);
}

// Initialize typing animation after a delay
setTimeout(initTypingAnimation, 1000);

// Copy to Clipboard Functionality
function initCopyToClipboard() {
    const copyButtons = document.querySelectorAll('.copy-btn');

    copyButtons.forEach(button => {
        button.addEventListener('click', function() {
            const codeBlock = this.parentElement.querySelector('code');
            const text = codeBlock.textContent;

            navigator.clipboard.writeText(text).then(function() {
                // Show success feedback
                const originalText = button.textContent;
                button.textContent = 'Copied!';
                button.style.background = 'var(--success-color)';

                setTimeout(function() {
                    button.textContent = originalText;
                    button.style.background = '';
                }, 2000);
            });
        });
    });
}

// Add copy buttons to code blocks
function addCopyButtons() {
    const codeBlocks = document.querySelectorAll('.code-content pre');

    codeBlocks.forEach(block => {
        const copyButton = document.createElement('button');
        copyButton.className = 'copy-btn';
        copyButton.innerHTML = '<i class="fas fa-copy"></i> Copy';
        copyButton.style.cssText = `
            position: absolute;
            top: 10px;
            right: 10px;
            background: var(--primary-color);
            color: white;
            border: none;
            padding: 5px 10px;
            border-radius: var(--radius);
            cursor: pointer;
            font-size: 0.8rem;
            transition: var(--transition);
        `;

        block.style.position = 'relative';
        block.appendChild(copyButton);
    });
}

// Initialize copy functionality
initCopyToClipboard();
addCopyButtons();

// Search Functionality
function initSearch() {
    const searchInput = document.getElementById('search-input');
    const searchResults = document.getElementById('search-results');

    if (!searchInput || !searchResults) return;

    searchInput.addEventListener('input', function() {
        const query = this.value.toLowerCase();
        const sections = document.querySelectorAll('.section');
        let results = [];

        sections.forEach(section => {
            const title = section.querySelector('.section-title');
            const content = section.textContent.toLowerCase();

            if (title && content.includes(query)) {
                results.push({
                    title: title.textContent,
                    id: section.id,
                    snippet: content.substring(0, 100) + '...'
                });
            }
        });

        displaySearchResults(results);
    });
}

function displaySearchResults(results) {
    const searchResults = document.getElementById('search-results');

    if (results.length === 0) {
        searchResults.innerHTML = '<p>No results found</p>';
        return;
    }

    let html = '';
    results.forEach(result => {
        html += `
            <div class="search-result">
                <a href="#${result.id}" class="search-result-link">
                    <h4>${result.title}</h4>
                    <p>${result.snippet}</p>
                </a>
            </div>
        `;
    });

    searchResults.innerHTML = html;
}

// Dark Mode Toggle (Optional)
function initDarkMode() {
    const darkModeToggle = document.getElementById('dark-mode-toggle');

    if (!darkModeToggle) return;

    darkModeToggle.addEventListener('click', function() {
        document.body.classList.toggle('dark-mode');

        const isDark = document.body.classList.contains('dark-mode');
        localStorage.setItem('darkMode', isDark);

        this.innerHTML = isDark ?
            '<i class="fas fa-sun"></i>' :
            '<i class="fas fa-moon"></i>';
    });

    // Load saved preference
    const savedDarkMode = localStorage.getItem('darkMode');
    if (savedDarkMode === 'true') {
        document.body.classList.add('dark-mode');
        darkModeToggle.innerHTML = '<i class="fas fa-sun"></i>';
    }
}

// Add dark mode styles
const darkModeStyles = document.createElement('style');
darkModeStyles.textContent = `
    .dark-mode {
        --primary-color: #60a5fa;
        --secondary-color: #94a3b8;
        --dark-color: #f8fafc;
        --light-color: #1f2937;
        --border-color: #374151;
        --text-primary: #f9fafb;
        --text-secondary: #d1d5db;
    }

    .dark-mode .navbar,
    .dark-mode .feature-card,
    .dark-mode .table-card,
    .dark-mode .guide-card,
    .dark-mode .workflow-step,
    .dark-mode .status-card,
    .dark-mode .tech-item,
    .dark-mode .arch-layer,
    .dark-mode .stat-card,
    .dark-mode .feature-item,
    .dark-mode .quick-start,
    .dark-mode .code-examples,
    .dark-mode .timeline-content {
        background: var(--dark-color);
        color: var(--text-primary);
        border-color: var(--border-color);
    }

    .dark-mode .section-alt {
        background: var(--light-color);
    }

    .dark-mode .code-content pre {
        background: var(--light-color);
        border-color: var(--border-color);
    }
`;
document.head.appendChild(darkModeStyles);

// Performance Monitoring
function initPerformanceMonitoring() {
    // Monitor page load time
    window.addEventListener('load', function() {
        const loadTime = performance.now();
        console.log(`Page loaded in ${loadTime.toFixed(2)} milliseconds`);

        // Send to analytics if available
        if (typeof gtag !== 'undefined') {
            gtag('event', 'page_load_time', {
                value: Math.round(loadTime)
            });
        }
    });

    // Monitor scroll depth
    let maxScroll = 0;
    window.addEventListener('scroll', function() {
        const scrollPercent = Math.round((window.scrollY / (document.documentElement.scrollHeight - window.innerHeight)) * 100);
        if (scrollPercent > maxScroll) {
            maxScroll = scrollPercent;
        }
    });

    // Send scroll depth on page unload
    window.addEventListener('beforeunload', function() {
        if (typeof gtag !== 'undefined' && maxScroll > 0) {
            gtag('event', 'scroll_depth', {
                value: maxScroll
            });
        }
    });
}

// Initialize performance monitoring
initPerformanceMonitoring();

// Error Handling
window.addEventListener('error', function(e) {
    console.error('JavaScript Error:', e.error);

    // Send to error tracking service if available
    if (typeof gtag !== 'undefined') {
        gtag('event', 'javascript_error', {
            error_message: e.error.message,
            error_file: e.filename,
            error_line: e.lineno
        });
    }
});

// Service Worker for PWA (Optional)
if ('serviceWorker' in navigator) {
    window.addEventListener('load', function() {
        // Register service worker for offline functionality
        // navigator.serviceWorker.register('/sw.js');
    });
}

// Accessibility Improvements
function initAccessibility() {
    // Add skip to main content link
    const skipLink = document.createElement('a');
    skipLink.href = '#overview';
    skipLink.className = 'skip-link';
    skipLink.textContent = 'Skip to main content';
    skipLink.style.cssText = `
        position: absolute;
        top: -40px;
        left: 6px;
        background: var(--primary-color);
        color: white;
        padding: 8px;
        text-decoration: none;
        border-radius: var(--radius);
        z-index: 1000;
        transition: var(--transition);
    `;

    document.body.insertBefore(skipLink, document.body.firstChild);

    // Focus management for skip link
    skipLink.addEventListener('focus', function() {
        this.style.top = '6px';
    });

    skipLink.addEventListener('blur', function() {
        this.style.top = '-40px';
    });

    // Add ARIA labels where needed
    const navToggle = document.querySelector('.nav-toggle');
    if (navToggle) {
        navToggle.setAttribute('aria-label', 'Toggle navigation menu');
        navToggle.setAttribute('aria-expanded', 'false');
    }
}

// Initialize accessibility features
initAccessibility();

// Export functions for testing
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        initNavigation,
        initScrollSpy,
        initCodeTabs,
        initSmoothScrolling,
        initScrollAnimations
    };
}