<x-layouts.app>
    <x-slot name="header">
        <div class="flex items-center justify-between">
            <h2 class="font-semibold text-xl text-gray-800 leading-tight">
                Audit Logs
            </h2>
            <div class="flex items-center space-x-4">
                <span class="px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm font-medium">
                    Audit Trail
                </span>
            </div>
        </div>
    </x-slot>

    <!-- Filters and Search -->
    <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100 mb-6">
        <div class="grid md:grid-cols-4 gap-4">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Date Range</label>
                <select id="dateRange" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    <option value="7">Last 7 days</option>
                    <option value="30" selected>Last 30 days</option>
                    <option value="90">Last 90 days</option>
                    <option value="365">Last year</option>
                </select>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Action Type</label>
                <select id="actionFilter" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    <option value="">All Actions</option>
                    <option value="login">Login</option>
                    <option value="logout">Logout</option>
                    <option value="create">Create</option>
                    <option value="update">Update</option>
                    <option value="delete">Delete</option>
                    <option value="view">View</option>
                </select>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Resource Type</label>
                <select id="resourceFilter" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    <option value="">All Resources</option>
                    <option value="user">User</option>
                    <option value="job">Job</option>
                    <option value="company">Company</option>
                    <option value="application">Application</option>
                </select>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Search</label>
                <input type="text" id="searchInput" placeholder="Search logs..." class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
            </div>
        </div>
    </div>

    <!-- Audit Logs Table -->
    <div class="bg-white rounded-2xl shadow-lg border border-gray-100">
        <div class="p-6 border-b border-gray-100">
            <div class="flex items-center justify-between">
                <h3 class="text-xl font-bold text-black">Activity Logs</h3>
                <div class="flex items-center space-x-2">
                    <button id="exportBtn" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition">
                        Export CSV
                    </button>
                    <button id="refreshBtn" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg font-medium transition">
                        Refresh
                    </button>
                </div>
            </div>
        </div>

        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Resource</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Description</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">IP Address</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Timestamp</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody id="auditLogsTable" class="bg-white divide-y divide-gray-200">
                    <!-- Logs will be loaded here -->
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <div class="px-6 py-4 border-t border-gray-100">
            <div class="flex items-center justify-between">
                <div class="text-sm text-gray-700">
                    Showing <span id="showingFrom">1</span> to <span id="showingTo">20</span> of <span id="totalLogs">0</span> entries
                </div>
                <div class="flex items-center space-x-2">
                    <button id="prevBtn" class="px-3 py-1 border border-gray-300 rounded text-sm hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">Previous</button>
                    <span id="currentPage" class="px-3 py-1 text-sm">1</span>
                    <button id="nextBtn" class="px-3 py-1 border border-gray-300 rounded text-sm hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">Next</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Log Details Modal -->
    <div id="logDetailsModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-3/4 lg:w-1/2 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-bold text-gray-900">Log Details</h3>
                    <button onclick="closeLogDetailsModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                        </svg>
                    </button>
                </div>
                <div id="logDetailsContent" class="space-y-4">
                    <!-- Details will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script>
        let currentPage = 1;
        let totalPages = 1;
        let currentFilters = {
            days: 30,
            action: '',
            resource: '',
            search: ''
        };

        document.addEventListener('DOMContentLoaded', function() {
            loadAuditLogs();

            // Event listeners
            document.getElementById('dateRange').addEventListener('change', function() {
                currentFilters.days = this.value;
                currentPage = 1;
                loadAuditLogs();
            });

            document.getElementById('actionFilter').addEventListener('change', function() {
                currentFilters.action = this.value;
                currentPage = 1;
                loadAuditLogs();
            });

            document.getElementById('resourceFilter').addEventListener('change', function() {
                currentFilters.resource = this.value;
                currentPage = 1;
                loadAuditLogs();
            });

            document.getElementById('searchInput').addEventListener('input', debounce(function() {
                currentFilters.search = this.value;
                currentPage = 1;
                loadAuditLogs();
            }, 500));

            document.getElementById('refreshBtn').addEventListener('click', function() {
                loadAuditLogs();
            });

            document.getElementById('exportBtn').addEventListener('click', function() {
                exportAuditLogs();
            });

            document.getElementById('prevBtn').addEventListener('click', function() {
                if (currentPage > 1) {
                    currentPage--;
                    loadAuditLogs();
                }
            });

            document.getElementById('nextBtn').addEventListener('click', function() {
                if (currentPage < totalPages) {
                    currentPage++;
                    loadAuditLogs();
                }
            });
        });

        function loadAuditLogs() {
            const params = new URLSearchParams({
                page: currentPage,
                days: currentFilters.days,
                action: currentFilters.action,
                resource: currentFilters.resource,
                search: currentFilters.search
            });

            fetch(`/admin/audit-logs/data?${params}`, {
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                renderAuditLogs(data.logs);
                updatePagination(data.pagination);
            })
            .catch(error => {
                console.error('Error loading audit logs:', error);
                showError('Failed to load audit logs');
            });
        }

        function renderAuditLogs(logs) {
            const tbody = document.getElementById('auditLogsTable');
            tbody.innerHTML = '';

            if (logs.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="7" class="px-6 py-8 text-center text-gray-500">
                            <svg class="w-12 h-12 mx-auto mb-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                            </svg>
                            <p>No audit logs found</p>
                        </td>
                    </tr>
                `;
                return;
            }

            logs.forEach(log => {
                const row = document.createElement('tr');
                row.className = 'hover:bg-gray-50';

                const actionClass = getActionClass(log.action);
                const resourceDisplay = log.resource_type ? `${log.resource_type.charAt(0).toUpperCase() + log.resource_type.slice(1)} #${log.resource_id}` : 'N/A';

                row.innerHTML = `
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="flex items-center">
                            <div class="w-8 h-8 bg-gradient-to-br from-blue-500 to-blue-600 rounded-full flex items-center justify-center mr-3">
                                <span class="text-white font-bold text-sm">${log.user ? log.user.name.charAt(0).toUpperCase() : 'U'}</span>
                            </div>
                            <div>
                                <div class="text-sm font-medium text-gray-900">${log.user ? log.user.name : 'Unknown User'}</div>
                                <div class="text-sm text-gray-500">${log.user ? log.user.email : ''}</div>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 text-xs font-medium rounded-full ${actionClass}">
                            ${log.action.charAt(0).toUpperCase() + log.action.slice(1)}
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        ${resourceDisplay}
                    </td>
                    <td class="px-6 py-4 text-sm text-gray-900 max-w-xs truncate" title="${log.description || ''}">
                        ${log.description || 'No description'}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        ${log.ip_address || 'N/A'}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        ${new Date(log.created_at).toLocaleString()}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <button onclick="viewLogDetails(${log.id})" class="text-blue-600 hover:text-blue-900">View Details</button>
                    </td>
                `;

                tbody.appendChild(row);
            });
        }

        function getActionClass(action) {
            const classes = {
                'login': 'bg-green-100 text-green-800',
                'logout': 'bg-gray-100 text-gray-800',
                'create': 'bg-blue-100 text-blue-800',
                'update': 'bg-yellow-100 text-yellow-800',
                'delete': 'bg-red-100 text-red-800',
                'view': 'bg-purple-100 text-purple-800'
            };
            return classes[action] || 'bg-gray-100 text-gray-800';
        }

        function updatePagination(pagination) {
            totalPages = pagination.last_page;
            document.getElementById('currentPage').textContent = pagination.current_page;
            document.getElementById('showingFrom').textContent = pagination.from || 0;
            document.getElementById('showingTo').textContent = pagination.to || 0;
            document.getElementById('totalLogs').textContent = pagination.total;

            document.getElementById('prevBtn').disabled = !pagination.prev_page_url;
            document.getElementById('nextBtn').disabled = !pagination.next_page_url;
        }

        function viewLogDetails(logId) {
            fetch(`/admin/audit-logs/${logId}`, {
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                showLogDetailsModal(data.log);
            })
            .catch(error => {
                console.error('Error loading log details:', error);
                showError('Failed to load log details');
            });
        }

        function showLogDetailsModal(log) {
            const content = document.getElementById('logDetailsContent');
            content.innerHTML = `
                <div class="grid md:grid-cols-2 gap-4">
                    <div>
                        <h4 class="font-semibold text-gray-900 mb-2">User Information</h4>
                        <div class="space-y-2 text-sm">
                            <p><span class="font-medium">Name:</span> ${log.user ? log.user.name : 'Unknown'}</p>
                            <p><span class="font-medium">Email:</span> ${log.user ? log.user.email : 'N/A'}</p>
                            <p><span class="font-medium">ID:</span> ${log.user_id}</p>
                        </div>
                    </div>
                    <div>
                        <h4 class="font-semibold text-gray-900 mb-2">Action Details</h4>
                        <div class="space-y-2 text-sm">
                            <p><span class="font-medium">Action:</span> ${log.action}</p>
                            <p><span class="font-medium">Resource:</span> ${log.resource_type || 'N/A'} ${log.resource_id ? '#' + log.resource_id : ''}</p>
                            <p><span class="font-medium">Timestamp:</span> ${new Date(log.created_at).toLocaleString()}</p>
                        </div>
                    </div>
                </div>
                <div class="mt-4">
                    <h4 class="font-semibold text-gray-900 mb-2">Description</h4>
                    <p class="text-sm text-gray-700">${log.description || 'No description available'}</p>
                </div>
                <div class="mt-4">
                    <h4 class="font-semibold text-gray-900 mb-2">Technical Details</h4>
                    <div class="bg-gray-50 p-3 rounded text-sm font-mono">
                        <p><span class="font-medium">IP Address:</span> ${log.ip_address || 'N/A'}</p>
                        <p><span class="font-medium">User Agent:</span> ${log.user_agent || 'N/A'}</p>
                        ${log.metadata ? `<p><span class="font-medium">Metadata:</span> <pre class="mt-1">${JSON.stringify(log.metadata, null, 2)}</pre></p>` : ''}
                    </div>
                </div>
            `;

            document.getElementById('logDetailsModal').classList.remove('hidden');
        }

        function closeLogDetailsModal() {
            document.getElementById('logDetailsModal').classList.add('hidden');
        }

        function exportAuditLogs() {
            const params = new URLSearchParams({
                days: currentFilters.days,
                action: currentFilters.action,
                resource: currentFilters.resource,
                search: currentFilters.search,
                export: 'csv'
            });

            window.open(`/admin/audit-logs/export?${params}`, '_blank');
        }

        function debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }

        function showError(message) {
            // Simple error notification - you can enhance this
            alert(message);
        }
    </script>
</x-layouts.app>