<x-layouts.app>
    <x-slot name="header">
        <div class="flex items-center justify-between">
            <h2 class="font-semibold text-xl text-gray-800 leading-tight">
                Bulk Email Campaigns
            </h2>
            <div class="flex items-center space-x-4">
                <span class="px-3 py-1 bg-green-100 text-green-800 rounded-full text-sm font-medium">
                    Email Marketing
                </span>
            </div>
        </div>
    </x-slot>

    <!-- Campaign Stats -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm font-medium">Total Campaigns</p>
                    <p class="text-3xl font-black text-black">{{ $stats['total_campaigns'] ?? 0 }}</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl flex items-center justify-center">
                    <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 4.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm font-medium">Emails Sent</p>
                    <p class="text-3xl font-black text-black">{{ number_format($stats['total_emails_sent'] ?? 0) }}</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-green-500 to-green-600 rounded-xl flex items-center justify-center">
                    <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm font-medium">Open Rate</p>
                    <p class="text-3xl font-black text-black">{{ $stats['average_open_rate'] ?? 0 }}%</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-yellow-500 to-yellow-600 rounded-xl flex items-center justify-center">
                    <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm font-medium">Click Rate</p>
                    <p class="text-3xl font-black text-black">{{ $stats['average_click_rate'] ?? 0 }}%</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-purple-500 to-purple-600 rounded-xl flex items-center justify-center">
                    <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"/>
                    </svg>
                </div>
            </div>
        </div>
    </div>

    <!-- Create Campaign Button -->
    <div class="mb-6">
        <button onclick="openCreateCampaignModal()" class="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-bold px-6 py-3 rounded-xl transition-all duration-300 shadow-lg hover:shadow-xl transform hover:scale-105">
            <svg class="w-5 h-5 inline mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"/>
            </svg>
            Create New Campaign
        </button>
    </div>

    <!-- Campaigns List -->
    <div class="bg-white rounded-2xl shadow-lg border border-gray-100">
        <div class="p-6 border-b border-gray-100">
            <div class="flex items-center justify-between">
                <h3 class="text-xl font-bold text-black">Email Campaigns</h3>
                <div class="flex items-center space-x-2">
                    <select id="statusFilter" class="px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                        <option value="">All Status</option>
                        <option value="draft">Draft</option>
                        <option value="scheduled">Scheduled</option>
                        <option value="sending">Sending</option>
                        <option value="sent">Sent</option>
                        <option value="failed">Failed</option>
                    </select>
                </div>
            </div>
        </div>

        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Campaign</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Recipients</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Sent</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Open Rate</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Click Rate</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody id="campaignsTable" class="bg-white divide-y divide-gray-200">
                    <!-- Campaigns will be loaded here -->
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <div class="px-6 py-4 border-t border-gray-100">
            <div class="flex items-center justify-between">
                <div class="text-sm text-gray-700">
                    Showing <span id="showingFrom">1</span> to <span id="showingTo">10</span> of <span id="totalCampaigns">0</span> campaigns
                </div>
                <div class="flex items-center space-x-2">
                    <button id="prevBtn" class="px-3 py-1 border border-gray-300 rounded text-sm hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">Previous</button>
                    <span id="currentPage" class="px-3 py-1 text-sm">1</span>
                    <button id="nextBtn" class="px-3 py-1 border border-gray-300 rounded text-sm hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">Next</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Create Campaign Modal -->
    <div id="createCampaignModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-4/5 lg:w-2/3 shadow-lg rounded-md bg-white max-h-screen overflow-y-auto">
            <div class="mt-3">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-bold text-gray-900">Create Email Campaign</h3>
                    <button onclick="closeCreateCampaignModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                        </svg>
                    </button>
                </div>

                <form id="createCampaignForm" class="space-y-6">
                    <div class="grid md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Campaign Name</label>
                            <input type="text" name="name" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Subject Line</label>
                            <input type="text" name="subject" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Recipient Groups</label>
                        <div class="space-y-2">
                            <label class="flex items-center">
                                <input type="checkbox" name="recipients[]" value="all_users" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                <span class="ml-2 text-sm">All Users</span>
                            </label>
                            <label class="flex items-center">
                                <input type="checkbox" name="recipients[]" value="job_seekers" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                <span class="ml-2 text-sm">Job Seekers</span>
                            </label>
                            <label class="flex items-center">
                                <input type="checkbox" name="recipients[]" value="employers" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                <span class="ml-2 text-sm">Employers</span>
                            </label>
                            <label class="flex items-center">
                                <input type="checkbox" name="recipients[]" value="inactive_users" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                <span class="ml-2 text-sm">Inactive Users (30+ days)</span>
                            </label>
                        </div>
                    </div>

                    <div class="grid md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Schedule Send</label>
                            <input type="datetime-local" name="scheduled_at" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                            <p class="text-xs text-gray-500 mt-1">Leave empty to send immediately</p>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Template</label>
                            <select name="template_id" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                <option value="">Select Template</option>
                                <option value="welcome">Welcome Email</option>
                                <option value="newsletter">Newsletter</option>
                                <option value="job_alert">Job Alert</option>
                                <option value="promotion">Promotion</option>
                            </select>
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Email Content</label>
                        <div class="border border-gray-300 rounded-lg">
                            <div class="p-4">
                                <textarea name="content" id="emailContent" rows="10" class="w-full border-0 focus:ring-0 resize-none" placeholder="Write your email content here..."></textarea>
                            </div>
                            <div class="border-t border-gray-300 p-3 bg-gray-50">
                                <div class="flex items-center justify-between">
                                    <div class="flex items-center space-x-2">
                                        <button type="button" onclick="insertPlaceholder('{user_name}')" class="px-3 py-1 text-xs bg-blue-100 text-blue-700 rounded hover:bg-blue-200">User Name</button>
                                        <button type="button" onclick="insertPlaceholder('{job_count}')" class="px-3 py-1 text-xs bg-green-100 text-green-700 rounded hover:bg-green-200">Job Count</button>
                                        <button type="button" onclick="insertPlaceholder('{company_name}')" class="px-3 py-1 text-xs bg-purple-100 text-purple-700 rounded hover:bg-purple-200">Company</button>
                                    </div>
                                    <span class="text-xs text-gray-500">Use placeholders for personalization</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="flex items-center justify-end space-x-4">
                        <button type="button" onclick="closeCreateCampaignModal()" class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">Cancel</button>
                        <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">Create Campaign</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        let currentPage = 1;
        let totalPages = 1;

        document.addEventListener('DOMContentLoaded', function() {
            loadCampaigns();

            document.getElementById('statusFilter').addEventListener('change', function() {
                currentPage = 1;
                loadCampaigns();
            });

            document.getElementById('prevBtn').addEventListener('click', function() {
                if (currentPage > 1) {
                    currentPage--;
                    loadCampaigns();
                }
            });

            document.getElementById('nextBtn').addEventListener('click', function() {
                if (currentPage < totalPages) {
                    currentPage++;
                    loadCampaigns();
                }
            });

            // Form submission
            document.getElementById('createCampaignForm').addEventListener('submit', function(e) {
                e.preventDefault();
                createCampaign();
            });
        });

        function loadCampaigns() {
            const status = document.getElementById('statusFilter').value;
            const params = new URLSearchParams({
                page: currentPage,
                status: status
            });

            fetch(`/admin/bulk-email/data?${params}`, {
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                renderCampaigns(data.campaigns);
                updatePagination(data.pagination);
            })
            .catch(error => {
                console.error('Error loading campaigns:', error);
            });
        }

        function renderCampaigns(campaigns) {
            const tbody = document.getElementById('campaignsTable');
            tbody.innerHTML = '';

            if (campaigns.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="7" class="px-6 py-8 text-center text-gray-500">
                            <svg class="w-12 h-12 mx-auto mb-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 4.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>
                            </svg>
                            <p>No email campaigns found</p>
                        </td>
                    </tr>
                `;
                return;
            }

            campaigns.forEach(campaign => {
                const statusClass = getStatusClass(campaign.status);
                const tbody = document.getElementById('campaignsTable');
                const row = document.createElement('tr');
                row.className = 'hover:bg-gray-50';

                row.innerHTML = `
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div>
                            <div class="text-sm font-medium text-gray-900">${campaign.name}</div>
                            <div class="text-sm text-gray-500">${campaign.subject}</div>
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        ${campaign.recipient_count || 0}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 text-xs font-medium rounded-full ${statusClass}">
                            ${campaign.status.charAt(0).toUpperCase() + campaign.status.slice(1)}
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        ${campaign.sent_at ? new Date(campaign.sent_at).toLocaleDateString() : 'Not sent'}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        ${campaign.open_rate || 0}%
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        ${campaign.click_rate || 0}%
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <div class="flex items-center space-x-2">
                            <button onclick="viewCampaign(${campaign.id})" class="text-blue-600 hover:text-blue-900">View</button>
                            ${campaign.status === 'draft' ? `<button onclick="editCampaign(${campaign.id})" class="text-yellow-600 hover:text-yellow-900">Edit</button>` : ''}
                            ${campaign.status === 'draft' || campaign.status === 'scheduled' ? `<button onclick="deleteCampaign(${campaign.id})" class="text-red-600 hover:text-red-900">Delete</button>` : ''}
                        </div>
                    </td>
                `;

                tbody.appendChild(row);
            });
        }

        function getStatusClass(status) {
            const classes = {
                'draft': 'bg-gray-100 text-gray-800',
                'scheduled': 'bg-blue-100 text-blue-800',
                'sending': 'bg-yellow-100 text-yellow-800',
                'sent': 'bg-green-100 text-green-800',
                'failed': 'bg-red-100 text-red-800'
            };
            return classes[status] || 'bg-gray-100 text-gray-800';
        }

        function updatePagination(pagination) {
            totalPages = pagination.last_page;
            document.getElementById('currentPage').textContent = pagination.current_page;
            document.getElementById('showingFrom').textContent = pagination.from || 0;
            document.getElementById('showingTo').textContent = pagination.to || 0;
            document.getElementById('totalCampaigns').textContent = pagination.total;

            document.getElementById('prevBtn').disabled = !pagination.prev_page_url;
            document.getElementById('nextBtn').disabled = !pagination.next_page_url;
        }

        function openCreateCampaignModal() {
            document.getElementById('createCampaignModal').classList.remove('hidden');
        }

        function closeCreateCampaignModal() {
            document.getElementById('createCampaignModal').classList.add('hidden');
            document.getElementById('createCampaignForm').reset();
        }

        function createCampaign() {
            const formData = new FormData(document.getElementById('createCampaignForm'));

            fetch('/admin/bulk-email/campaigns', {
                method: 'POST',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    closeCreateCampaignModal();
                    loadCampaigns();
                    showSuccess('Campaign created successfully');
                } else {
                    showError(data.message || 'Failed to create campaign');
                }
            })
            .catch(error => {
                console.error('Error creating campaign:', error);
                showError('Failed to create campaign');
            });
        }

        function insertPlaceholder(placeholder) {
            const textarea = document.getElementById('emailContent');
            const start = textarea.selectionStart;
            const end = textarea.selectionEnd;
            const text = textarea.value;
            const before = text.substring(0, start);
            const after = text.substring(end, text.length);

            textarea.value = before + placeholder + after;
            textarea.selectionStart = textarea.selectionEnd = start + placeholder.length;
            textarea.focus();
        }

        function viewCampaign(id) {
            // Redirect to campaign details page
            window.location.href = `/admin/bulk-email/campaigns/${id}`;
        }

        function editCampaign(id) {
            // Redirect to campaign edit page
            window.location.href = `/admin/bulk-email/campaigns/${id}/edit`;
        }

        function deleteCampaign(id) {
            if (confirm('Are you sure you want to delete this campaign?')) {
                fetch(`/admin/bulk-email/campaigns/${id}`, {
                    method: 'DELETE',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        loadCampaigns();
                        showSuccess('Campaign deleted successfully');
                    } else {
                        showError(data.message || 'Failed to delete campaign');
                    }
                })
                .catch(error => {
                    console.error('Error deleting campaign:', error);
                    showError('Failed to delete campaign');
                });
            }
        }

        function showSuccess(message) {
            // Simple success notification
            alert(message);
        }

        function showError(message) {
            // Simple error notification
            alert(message);
        }
    </script>
</x-layouts.app>