<x-layouts.app>
    <x-slot name="header">
        <div class="flex items-center justify-between">
            <h2 class="font-semibold text-xl text-gray-800 leading-tight">
                {{ __('Browse Jobs') }}
            </h2>
            <a href="/" class="inline-flex items-center text-gray-600 hover:text-yellow-600 transition-colors">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
                </svg>
                Back to Home
            </a>
        </div>
    </x-slot>

    <!-- Hero Section -->
    <div class="bg-gradient-to-br from-yellow-400 via-yellow-500 to-orange-500 text-black">
        <div class="max-w-7xl mx-auto px-8 py-16">
            <div class="text-center">
                <h1 class="text-5xl md:text-6xl font-black mb-6">
                    Find Your Dream Job
                </h1>
                <p class="text-xl md:text-2xl text-black/80 max-w-3xl mx-auto mb-8">
                    Discover amazing opportunities across Cameroon with our smart job search platform.
                </p>
            </div>
        </div>
    </div>

    <!-- Search Section -->
    <div class="bg-white shadow-lg -mt-8 relative z-10 mx-4 sm:mx-8 rounded-2xl border border-gray-100">
        <div class="p-8">
            <div class="max-w-6xl mx-auto">

                    <!-- Advanced Search Form -->
                    <form method="GET" action="{{ route('jobs.index') }}" class="space-y-6" id="searchForm">
                        <!-- Primary Search Row -->
                        <div class="bg-white/80 backdrop-blur-sm rounded-2xl p-6 shadow-xl border border-white/50">
                            <div class="grid md:grid-cols-5 gap-6">
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                        <svg class="w-4 h-4 mr-2 text-yellow-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                                        </svg>
                                        Keywords
                                    </label>
                                    <div class="relative group">
                                        <input type="text" name="q" value="{{ request('q') }}"
                                               placeholder="Job title, skills, or company"
                                               class="w-full pl-12 pr-4 py-4 bg-white border-2 border-gray-200 rounded-xl text-gray-900 placeholder-gray-500 focus:outline-none focus:ring-4 focus:ring-yellow-400/20 focus:border-yellow-400 transition-all duration-300 text-lg shadow-sm group-hover:shadow-md">
                                        <div class="absolute left-4 top-4 w-6 h-6 bg-gradient-to-br from-yellow-400 to-yellow-500 rounded-lg flex items-center justify-center">
                                            <svg class="w-4 h-4 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                                            </svg>
                                        </div>
                                    </div>
                                </div>
                                <div>
                                    <label class="block text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                        <svg class="w-4 h-4 mr-2 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"/>
                                        </svg>
                                        Location
                                    </label>
                                    <select name="location" class="w-full px-4 py-4 bg-white border-2 border-gray-200 rounded-xl text-gray-900 focus:outline-none focus:ring-4 focus:ring-brand-green/20 focus:border-brand-green transition-all duration-300 shadow-sm appearance-none">
                                        <option value="">All Locations</option>
                                        <option value="douala" {{ request('location') == 'douala' ? 'selected' : '' }}>Douala</option>
                                        <option value="yaounde" {{ request('location') == 'yaounde' ? 'selected' : '' }}>Yaoundé</option>
                                        <option value="limbe" {{ request('location') == 'limbe' ? 'selected' : '' }}>Limbé</option>
                                        <option value="bafoussam" {{ request('location') == 'bafoussam' ? 'selected' : '' }}>Bafoussam</option>
                                        <option value="bamenda" {{ request('location') == 'bamenda' ? 'selected' : '' }}>Bamenda</option>
                                        <option value="garoua" {{ request('location') == 'garoua' ? 'selected' : '' }}>Garoua</option>
                                        <option value="maroua" {{ request('location') == 'maroua' ? 'selected' : '' }}>Maroua</option>
                                        <option value="ngaoundere" {{ request('location') == 'ngaoundere' ? 'selected' : '' }}>Ngaoundéré</option>
                                        <option value="remote" {{ request('location') == 'remote' ? 'selected' : '' }}>Remote</option>
                                    </select>
                                </div>
                                <div>
                                    <label class="block text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                        <svg class="w-4 h-4 mr-2 text-purple-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"/>
                                        </svg>
                                        Category
                                    </label>
                                    <select name="sector" class="w-full px-4 py-4 bg-white border-2 border-gray-200 rounded-xl text-gray-900 focus:outline-none focus:ring-4 focus:ring-brand-purple/20 focus:border-brand-purple transition-all duration-300 shadow-sm appearance-none">
                                        <option value="">All Categories</option>
                                        <option value="technology" {{ request('sector') == 'technology' ? 'selected' : '' }}>Technology</option>
                                        <option value="finance" {{ request('sector') == 'finance' ? 'selected' : '' }}>Finance & Banking</option>
                                        <option value="healthcare" {{ request('sector') == 'healthcare' ? 'selected' : '' }}>Healthcare</option>
                                        <option value="education" {{ request('sector') == 'education' ? 'selected' : '' }}>Education</option>
                                        <option value="retail" {{ request('sector') == 'retail' ? 'selected' : '' }}>Retail & Commerce</option>
                                        <option value="logistics" {{ request('sector') == 'logistics' ? 'selected' : '' }}>Logistics</option>
                                        <option value="manufacturing" {{ request('sector') == 'manufacturing' ? 'selected' : '' }}>Manufacturing</option>
                                        <option value="construction" {{ request('sector') == 'construction' ? 'selected' : '' }}>Construction</option>
                                        <option value="agriculture" {{ request('sector') == 'agriculture' ? 'selected' : '' }}>Agriculture</option>
                                        <option value="energy" {{ request('sector') == 'energy' ? 'selected' : '' }}>Energy & Utilities</option>
                                        <option value="government" {{ request('sector') == 'government' ? 'selected' : '' }}>Government</option>
                                    </select>
                                </div>
                                <div>
                                    <label class="block text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                        <svg class="w-4 h-4 mr-2 text-pink-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                        </svg>
                                        Experience
                                    </label>
                                    <select name="experience_level" class="w-full px-4 py-4 bg-white border-2 border-gray-200 rounded-xl text-gray-900 focus:outline-none focus:ring-4 focus:ring-brand-pink/20 focus:border-brand-pink transition-all duration-300 shadow-sm appearance-none">
                                        <option value="">Any Experience</option>
                                        <option value="entry" {{ request('experience_level') == 'entry' ? 'selected' : '' }}>Entry Level (0-2 years)</option>
                                        <option value="mid" {{ request('experience_level') == 'mid' ? 'selected' : '' }}>Mid Level (3-5 years)</option>
                                        <option value="senior" {{ request('experience_level') == 'senior' ? 'selected' : '' }}>Senior Level (5+ years)</option>
                                        <option value="executive" {{ request('experience_level') == 'executive' ? 'selected' : '' }}>Executive Level</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <!-- Advanced Filters Row -->
                        <div class="bg-white/60 backdrop-blur-sm rounded-2xl p-6 shadow-lg border border-white/50">
                            <div class="grid md:grid-cols-4 gap-6">
                                <div>
                                    <label class="block text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                        <svg class="w-4 h-4 mr-2 text-indigo-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 13.255A23.931 23.931 0 0112 15c-3.183 0-6.22-.62-9-1.745M16 6V4a2 2 0 00-2-2h-4a2 2 0 00-2 2v2m8 0V8a2 2 0 01-2 2H8a2 2 0 01-2-2V6m8 0H8"/>
                                        </svg>
                                        Employment Type
                                    </label>
                                    <select name="employment_type" class="w-full px-4 py-4 bg-white border-2 border-gray-200 rounded-xl text-gray-900 focus:outline-none focus:ring-4 focus:ring-brand-indigo/20 focus:border-brand-indigo transition-all duration-300 shadow-sm appearance-none">
                                        <option value="">Any Type</option>
                                        <option value="full-time" {{ request('employment_type') == 'full-time' ? 'selected' : '' }}>Full-time</option>
                                        <option value="part-time" {{ request('employment_type') == 'part-time' ? 'selected' : '' }}>Part-time</option>
                                        <option value="contract" {{ request('employment_type') == 'contract' ? 'selected' : '' }}>Contract</option>
                                        <option value="freelance" {{ request('employment_type') == 'freelance' ? 'selected' : '' }}>Freelance</option>
                                        <option value="internship" {{ request('employment_type') == 'internship' ? 'selected' : '' }}>Internship</option>
                                    </select>
                                </div>
                                <div>
                                    <label class="block text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                        <svg class="w-4 h-4 mr-2 text-emerald-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"/>
                                        </svg>
                                        Min Salary (XAF)
                                    </label>
                                    <input type="number" name="salary_min" value="{{ request('salary_min') }}"
                                           placeholder="e.g. 50000"
                                           class="w-full px-4 py-4 bg-white border-2 border-gray-200 rounded-xl text-gray-900 placeholder-gray-500 focus:outline-none focus:ring-4 focus:ring-brand-emerald/20 focus:border-brand-emerald transition-all duration-300 shadow-sm">
                                </div>
                                <div>
                                    <label class="block text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                        <svg class="w-4 h-4 mr-2 text-orange-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"/>
                                        </svg>
                                        Max Salary (XAF)
                                    </label>
                                    <input type="number" name="salary_max" value="{{ request('salary_max') }}"
                                           placeholder="e.g. 200000"
                                           class="w-full px-4 py-4 bg-white border-2 border-gray-200 rounded-xl text-gray-900 placeholder-gray-500 focus:outline-none focus:ring-4 focus:ring-orange-500/20 focus:border-orange-500 transition-all duration-300 shadow-sm">
                                </div>
                                <div>
                                    <label class="block text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                        <svg class="w-4 h-4 mr-2 text-cyan-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                        </svg>
                                        Posted Within
                                    </label>
                                    <select name="posted_within" class="w-full px-4 py-4 bg-white border-2 border-gray-200 rounded-xl text-gray-900 focus:outline-none focus:ring-4 focus:ring-brand-cyan/20 focus:border-brand-cyan transition-all duration-300 shadow-sm appearance-none">
                                        <option value="">Any Time</option>
                                        <option value="1" {{ request('posted_within') == '1' ? 'selected' : '' }}>Last 24 hours</option>
                                        <option value="3" {{ request('posted_within') == '3' ? 'selected' : '' }}>Last 3 days</option>
                                        <option value="7" {{ request('posted_within') == '7' ? 'selected' : '' }}>Last week</option>
                                        <option value="30" {{ request('posted_within') == '30' ? 'selected' : '' }}>Last month</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <!-- Filter Options & Actions -->
                        <div class="bg-white/40 backdrop-blur-sm rounded-2xl p-6 shadow-lg border border-white/50">
                            <div class="flex flex-col lg:flex-row lg:items-center justify-between gap-6">
                                <div class="flex flex-wrap items-center gap-6">
                                    <div class="flex items-center group">
                                        <input type="checkbox" id="remote" name="remote" value="1" {{ request('remote') ? 'checked' : '' }}
                                               class="w-5 h-5 rounded border-2 border-gray-300 text-yellow-600 focus:ring-yellow-500 focus:ring-2 transition-all duration-200 group-hover:border-yellow-400">
                                        <label for="remote" class="ml-3 text-gray-700 font-medium group-hover:text-yellow-600 transition-colors cursor-pointer">Remote jobs only</label>
                                    </div>
                                    <div class="flex items-center group">
                                        <input type="checkbox" id="featured" name="featured" value="1" {{ request('featured') ? 'checked' : '' }}
                                               class="w-5 h-5 rounded border-2 border-gray-300 text-purple-600 focus:ring-purple-500 focus:ring-2 transition-all duration-200 group-hover:border-purple-400">
                                        <label for="featured" class="ml-3 text-gray-700 font-medium group-hover:text-purple-600 transition-colors cursor-pointer">Featured jobs</label>
                                    </div>
                                    <div class="flex items-center group">
                                        <input type="checkbox" id="with_salary" name="with_salary" value="1" {{ request('with_salary') ? 'checked' : '' }}
                                               class="w-5 h-5 rounded border-2 border-gray-300 text-green-600 focus:ring-green-500 focus:ring-2 transition-all duration-200 group-hover:border-green-400">
                                        <label for="with_salary" class="ml-3 text-gray-700 font-medium group-hover:text-green-600 transition-colors cursor-pointer">Jobs with salary info</label>
                                    </div>
                                </div>

                                <div class="flex gap-4">
                                    <button type="submit" class="bg-gradient-to-r from-yellow-400 to-yellow-500 hover:from-yellow-500 hover:to-yellow-600 text-black font-bold px-8 py-4 rounded-xl transition-all duration-300 shadow-xl hover:shadow-2xl transform hover:scale-105 hover:-translate-y-1 flex items-center">
                                        <svg class="w-6 h-6 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                                        </svg>
                                        Search Jobs
                                    </button>
                                    <a href="{{ route('jobs.index') }}" class="bg-white/80 hover:bg-white text-gray-700 hover:text-gray-900 px-6 py-4 rounded-xl transition-all duration-300 shadow-lg hover:shadow-xl transform hover:scale-105 flex items-center backdrop-blur-sm border border-gray-200">
                                        <svg class="w-6 h-6 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/>
                                        </svg>
                                        Clear All
                                    </a>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Results Summary & Controls -->
    <div class="bg-white rounded-2xl shadow-lg p-6 mb-6">
        <div class="flex flex-col lg:flex-row lg:items-center justify-between gap-4">
            <div class="flex-1">
                <div class="text-gray-900 text-lg font-semibold mb-2">
                    <span class="text-2xl">{{ $jobs->total() }}</span> jobs found
                    @if(request('q'))
                    for "<span class="text-brand-gold font-bold">{{ request('q') }}</span>"
                    @endif
                </div>
                <div class="flex flex-wrap items-center gap-4 text-sm text-gray-600">
                    @if(request('location'))
                        <span class="flex items-center">
                            <svg class="w-4 h-4 mr-1 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                            </svg>
                            {{ ucfirst(request('location')) }}
                        </span>
                    @endif
                    @if(request('sector'))
                        <span class="flex items-center">
                            <svg class="w-4 h-4 mr-1 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"/>
                            </svg>
                            {{ ucfirst(request('sector')) }}
                        </span>
                    @endif
                    @if(request('remote'))
                        <span class="flex items-center">
                            <svg class="w-4 h-4 mr-1 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9.75 17L9 20l-1 1h8l-1-1-.75-3M3 13h18M5 17h14a2 2 0 002-2V5a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>
                            </svg>
                            Remote Work
                        </span>
                    @endif
                    @if(request('featured'))
                        <span class="flex items-center">
                            <svg class="w-4 h-4 mr-1 text-yellow-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.519 4.674a1 1 0 00.95.69h4.915c.969 0 1.371 1.24.588 1.81l-3.976 2.888a1 1 0 00-.363 1.118l1.518 4.674c.3.922-.755 1.688-1.538 1.118l-3.976-2.888a1 1 0 00-1.176 0l-3.976 2.888c-.783.57-1.838-.197-1.538-1.118l1.518-4.674a1 1 0 00-.363-1.118l-3.976-2.888c-.784-.57-.38-1.81.588-1.81h4.914a1 1 0 00.951-.69l1.519-4.674z"/>
                            </svg>
                            Featured Only
                        </span>
                    @endif
                </div>
            </div>

            <div class="flex items-center gap-4">
                <!-- View Toggle -->
                <div class="flex items-center bg-gray-100 rounded-lg p-1">
                    <button class="p-2 rounded-md text-gray-600 hover:text-gray-900 hover:bg-white transition {{ request('view', 'list') == 'list' ? 'bg-white text-gray-900 shadow-sm' : '' }}"
                            onclick="setView('list')">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 10h16M4 14h16M4 18h16"/>
                        </svg>
                    </button>
                    <button class="p-2 rounded-md text-gray-600 hover:text-gray-900 hover:bg-white transition {{ request('view') == 'grid' ? 'bg-white text-gray-900 shadow-sm' : '' }}"
                            onclick="setView('grid')">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2V6zM14 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V6zM4 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2v-2zM14 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2v-2z"/>
                        </svg>
                    </button>
                </div>

                <!-- Sort Options -->
                <form method="GET" action="{{ route('jobs.index') }}" class="flex items-center gap-2">
                    @foreach(request()->all() as $key => $value)
                        @if($key !== 'sort' && $key !== 'view')
                            <input type="hidden" name="{{ $key }}" value="{{ $value }}">
                        @endif
                    @endforeach
                    <select name="sort" onchange="this.form.submit()" class="border border-gray-300 rounded-lg px-4 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-brand-gold focus:border-transparent bg-white">
                        <option value="newest" {{ request('sort', 'newest') == 'newest' ? 'selected' : '' }}>Newest First</option>
                        <option value="oldest" {{ request('sort') == 'oldest' ? 'selected' : '' }}>Oldest First</option>
                        <option value="salary_high" {{ request('sort') == 'salary_high' ? 'selected' : '' }}>Highest Salary</option>
                        <option value="salary_low" {{ request('sort') == 'salary_low' ? 'selected' : '' }}>Lowest Salary</option>
                        <option value="relevance" {{ request('sort') == 'relevance' ? 'selected' : '' }}>Most Relevant</option>
                    </select>
                </form>

                <!-- Save Search -->
                @auth
                <button class="bg-gray-100 hover:bg-gray-200 text-gray-700 px-4 py-2 rounded-lg text-sm font-medium transition flex items-center"
                        onclick="saveSearch()">
                    <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                    </svg>
                    Save Search
                </button>
                @endauth
            </div>
        </div>

        <!-- Active Filters -->
        @if(request()->hasAny(['q', 'location', 'sector', 'experience_level', 'employment_type', 'salary_min', 'salary_max', 'posted_within', 'remote', 'featured', 'with_salary']))
        <div class="mt-4 pt-4 border-t border-gray-200">
            <div class="flex flex-wrap items-center gap-2">
                <span class="text-sm font-medium text-gray-700 mr-2">Active filters:</span>
                @if(request('q'))
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm bg-yellow-100 text-yellow-800">
                        "{{ request('q') }}"
                        <a href="{{ route('jobs.index', array_merge(request()->all(), ['q' => null])) }}" class="ml-2 text-yellow-600 hover:text-yellow-800">
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                            </svg>
                        </a>
                    </span>
                @endif
                @if(request('location'))
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm bg-yellow-100 text-yellow-800">
                        {{ ucfirst(request('location')) }}
                        <a href="{{ route('jobs.index', array_merge(request()->all(), ['location' => null])) }}" class="ml-2 text-yellow-600 hover:text-yellow-800">
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                            </svg>
                        </a>
                    </span>
                @endif
                @if(request('sector'))
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm bg-gray-100 text-gray-800">
                        {{ ucfirst(request('sector')) }}
                        <a href="{{ route('jobs.index', array_merge(request()->all(), ['sector' => null])) }}" class="ml-2 text-gray-600 hover:text-gray-800">
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                            </svg>
                        </a>
                    </span>
                @endif
                @if(request('remote'))
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm bg-gray-100 text-gray-800">
                        Remote Work
                        <a href="{{ route('jobs.index', array_merge(request()->all(), ['remote' => null])) }}" class="ml-2 text-gray-600 hover:text-gray-800">
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                            </svg>
                        </a>
                    </span>
                @endif
                @if(request('featured'))
                <span class="inline-flex items-center px-3 py-1 rounded-full text-sm bg-brand-gold/10 text-brand-gold">
                    Featured Only
                    <a href="{{ route('jobs.index', array_merge(request()->all(), ['featured' => null])) }}" class="ml-2 text-brand-gold hover:text-brand-amber">
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                            </svg>
                        </a>
                    </span>
                @endif
                <a href="{{ route('jobs.index') }}" class="text-sm text-gray-500 hover:text-gray-700 underline ml-2">
                    Clear all filters
                </a>
            </div>
        </div>
        @endif
    </div>

    <!-- Job Listings -->
    <div class="max-w-7xl mx-auto">
        <!-- Simplified Grid: 3 columns on desktop, 1 on mobile -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            @forelse($jobs as $job)
            <!-- Simplified Job Card -->
            <div class="bg-white rounded-xl shadow-md border border-gray-100 overflow-hidden hover:shadow-lg transition-all duration-200 flex flex-col h-full">
                <!-- Card Header -->
                <div class="p-4 bg-gray-50 border-b border-gray-100">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-3">
                            <!-- Company Logo -->
                            <div class="w-12 h-12 rounded-full bg-gradient-to-br from-yellow-400 to-yellow-500 flex items-center justify-center flex-shrink-0">
                                @if($job->company && $job->company->logo)
                                    <img src="{{ asset('storage/' . $job->company->logo) }}" alt="{{ $job->company->name }}" class="w-12 h-12 rounded-full object-cover">
                                @else
                                    <span class="text-black font-bold text-lg">{{ substr($job->company ? $job->company->name : 'C', 0, 1) }}</span>
                                @endif
                            </div>
                            <div>
                                <h4 class="font-semibold text-gray-900 hover:text-yellow-600 cursor-pointer" onclick="window.location.href='{{ route('companies.show', $job->company) }}'">
                                    {{ $job->company ? $job->company->name : 'Company Name' }}
                                </h4>
                                <p class="text-sm text-gray-600">{{ $job->location }} • {{ $job->created_at->diffForHumans() }}</p>
                            </div>
                        </div>
                        @if($job->featured)
                            <div class="bg-yellow-100 text-yellow-800 px-2 py-1 rounded-full text-xs font-semibold">
                                Featured
                            </div>
                        @endif
                    </div>
                </div>

                <!-- Card Body -->
                <div class="p-4 flex-1 flex flex-col">
                    <!-- Job Title -->
                    <h3 class="text-lg font-bold text-gray-900 mb-2 line-clamp-2">
                        {{ $job->title }}
                    </h3>

                    <!-- Job Description -->
                    <p class="text-gray-700 mb-4 text-sm leading-relaxed line-clamp-3">
                        {{ Str::limit($job->description, 100) }}
                    </p>

                    <!-- Job Tags -->
                    <div class="flex flex-wrap gap-2 mb-4">
                        <span class="px-3 py-1 bg-blue-100 text-blue-800 text-xs rounded-full font-medium">
                            {{ $job->employment_type ?? 'Full-time' }}
                        </span>
                        @if($job->salary_max || $job->salary_min)
                        <span class="px-3 py-1 bg-green-100 text-green-800 text-xs rounded-full font-medium">
                            XAF {{ number_format($job->salary_max ?? $job->salary_min ?? 0) }}
                        </span>
                        @endif
                        @if($job->experience_level)
                        <span class="px-3 py-1 bg-purple-100 text-purple-800 text-xs rounded-full font-medium">
                            {{ ucfirst($job->experience_level) }}
                        </span>
                        @endif
                    </div>

                    <!-- Skills -->
                    @if($job->skills)
                    <div class="flex flex-wrap gap-1 mb-4">
                        @foreach(array_slice(explode(',', $job->skills), 0, 3) as $skill)
                        <span class="px-2 py-1 bg-yellow-100 text-yellow-700 text-xs rounded font-medium">
                            {{ trim($skill) }}
                        </span>
                        @endforeach
                        @if(count(explode(',', $job->skills)) > 3)
                        <span class="px-2 py-1 bg-gray-100 text-gray-600 text-xs rounded">
                            +{{ count(explode(',', $job->skills)) - 3 }}
                        </span>
                        @endif
                    </div>
                    @endif

                    <!-- Spacer -->
                    <div class="flex-1"></div>

                    <!-- Card Footer -->
                    <div class="pt-4 border-t border-gray-100">
                        <div class="flex items-center justify-between">
                            <a href="{{ route('jobs.show', $job) }}" class="text-yellow-600 hover:text-yellow-800 text-sm font-medium">
                                View Details →
                            </a>
                            @auth
                                @if(auth()->user()->hasAppliedToJob($job))
                                    <button disabled class="bg-green-600 text-white px-4 py-2 rounded-lg text-sm font-semibold">
                                        Applied
                                    </button>
                                @else
                                    <button onclick="quickApply({{ $job->id }})" class="bg-yellow-400 hover:bg-yellow-500 text-black px-4 py-2 rounded-lg text-sm font-semibold transition-colors">
                                        Apply
                                    </button>
                                @endif
                            @else
                                <a href="{{ route('login') }}" class="bg-yellow-400 hover:bg-yellow-500 text-black px-4 py-2 rounded-lg text-sm font-semibold transition-colors">
                                    Login
                                </a>
                            @endauth
                        </div>
                    </div>
                </div>
            </div>
        @empty
        <div class="bg-white rounded-3xl shadow-2xl p-16 text-center relative overflow-hidden">
            <!-- Background Pattern -->
            <div class="absolute inset-0 bg-gradient-to-br from-gray-50 to-blue-50 opacity-50"></div>
            <div class="relative">
                <div class="w-32 h-32 bg-gradient-to-br from-gray-200 to-gray-300 rounded-full flex items-center justify-center mx-auto mb-8">
                    <svg class="w-16 h-16 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                    </svg>
                </div>
                <h3 class="text-3xl font-bold text-gray-900 mb-4">No jobs found</h3>
                <p class="text-xl text-gray-600 mb-8 max-w-md mx-auto">Try adjusting your search criteria or check back later for new opportunities.</p>
                <div class="flex gap-4 justify-center">
                    <a href="{{ route('jobs.index') }}" class="bg-gradient-to-r from-yellow-400 to-yellow-500 hover:from-yellow-500 hover:to-yellow-600 text-black font-bold px-8 py-4 rounded-xl transition-all duration-300 shadow-xl hover:shadow-2xl transform hover:scale-105 hover:-translate-y-1">
                        Clear Filters
                    </a>
                    <a href="{{ route('jobs.create') }}" class="bg-white border-2 border-gray-300 hover:border-black text-gray-700 hover:text-black font-bold px-8 py-4 rounded-xl transition-all duration-300 shadow-lg hover:shadow-xl transform hover:scale-105">
                        Post a Job
                    </a>
                </div>
            </div>
        </div>
        @endforelse
    </div>

    <!-- Pagination -->
    @if($jobs->hasPages())
    <div class="mt-8">
        {{ $jobs->appends(request()->query())->links() }}
    </div>
    @endif

    <!-- Quick Apply Modal (if needed) -->
    <div id="quickApplyModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-2xl max-w-md w-full p-6">
                <h3 class="text-xl font-bold text-black mb-4">Quick Apply</h3>
                <p class="text-gray-600 mb-6">Apply with your existing profile information.</p>
                <div class="flex gap-3">
                    <button class="flex-1 bg-gradient-to-r from-yellow-400 to-yellow-500 hover:from-yellow-500 hover:to-yellow-600 text-black font-bold py-3 px-4 rounded-lg transition">
                        Apply Now
                    </button>
                    <button onclick="closeModal()" class="px-4 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition">
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>
</x-layouts.app>

<script>
function closeModal() {
    document.getElementById('quickApplyModal').classList.add('hidden');
}

function setView(viewType) {
    const url = new URL(window.location);
    url.searchParams.set('view', viewType);
    window.location.href = url.toString();
}

function saveJob(jobId) {
    @auth
        fetch(`/api/jobs/${jobId}/save`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Toggle heart icon
                const button = event.target.closest('button');
                const icon = button.querySelector('svg');
                if (data.saved) {
                    icon.classList.add('fill-current', 'text-red-500');
                    showNotification('Job saved successfully!', 'success');
                } else {
                    icon.classList.remove('fill-current', 'text-red-500');
                    showNotification('Job removed from saved jobs', 'info');
                }
            } else {
                showNotification('Failed to save job. Please try again.', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Failed to save job. Please try again.', 'error');
        });
    @else
        window.location.href = '{{ route("login") }}';
    @endauth
}

function shareJob(jobId, jobTitle) {
    const url = `${window.location.origin}/jobs/${jobId}`;

    if (navigator.share) {
        navigator.share({
            title: jobTitle,
            text: `Check out this job opportunity: ${jobTitle}`,
            url: url
        }).catch(() => {
            // Fallback to clipboard if share fails
            copyToClipboard(url);
        });
    } else {
        // Fallback: copy to clipboard
        copyToClipboard(url);
    }
}

function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        showNotification('Job link copied to clipboard!', 'success');
    }).catch(() => {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = text;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        showNotification('Job link copied to clipboard!', 'success');
    });
}

function visitCompanyProfile(companyId) {
    if (companyId) {
        window.location.href = `/companies/${companyId}`;
    }
}

function toggleJobOptions(jobId) {
    const options = document.getElementById(`job-options-${jobId}`);
    // Close other open options
    document.querySelectorAll('[id^="job-options-"]').forEach(el => {
        if (el.id !== `job-options-${jobId}`) {
            el.classList.add('hidden');
        }
    });
    options.classList.toggle('hidden');
}

function reportJob(jobId) {
    if (confirm('Are you sure you want to report this job?')) {
        fetch(`/api/jobs/${jobId}/report`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ reason: 'inappropriate_content' })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('Job reported successfully', 'success');
                document.getElementById(`job-options-${jobId}`).classList.add('hidden');
            } else {
                showNotification('Failed to report job', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Failed to report job', 'error');
        });
    }
}

function hideJob(jobId) {
    if (confirm('Are you sure you want to hide this job?')) {
        // Hide the job card
        const jobCard = document.querySelector(`[data-job-id="${jobId}"]`);
        if (jobCard) {
            jobCard.closest('.bg-white').style.display = 'none';
        }
        showNotification('Job hidden from your feed', 'info');
        document.getElementById(`job-options-${jobId}`).classList.add('hidden');
    }
}

function likeJob(jobId) {
    @auth
        const button = document.querySelector(`.like-btn[data-job-id="${jobId}"]`);
        const icon = button.querySelector('svg');
        const likesCount = document.getElementById(`likes-count-${jobId}`);

        fetch(`/api/v1/jobs/${jobId}/like`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (data.liked) {
                    icon.classList.add('fill-current', 'text-red-500');
                    if (likesCount) likesCount.textContent = parseInt(likesCount.textContent) + 1;
                } else {
                    icon.classList.remove('fill-current', 'text-red-500');
                    if (likesCount) likesCount.textContent = Math.max(0, parseInt(likesCount.textContent) - 1);
                }
                showNotification(data.liked ? 'Job liked!' : 'Job unliked', 'success');
            } else {
                showNotification('Failed to like job', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Failed to like job', 'error');
        });
    @else
        window.location.href = '{{ route("login") }}';
    @endauth
}

function showComments(jobId) {
    const commentsSection = document.getElementById(`comments-${jobId}`);
    const isHidden = commentsSection.classList.contains('hidden');

    if (isHidden) {
        // Load comments when showing
        loadComments(jobId);
    }

    commentsSection.classList.toggle('hidden');
}

function loadComments(jobId) {
    fetch(`/api/v1/jobs/${jobId}/comments`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                renderComments(jobId, data.data);
            }
        })
        .catch(error => {
            console.error('Error loading comments:', error);
        });
}

function renderComments(jobId, comments) {
    const commentsList = document.getElementById(`comments-list-${jobId}`);
    const commentsCount = document.getElementById(`comments-count-${jobId}`);

    if (commentsCount) {
        commentsCount.textContent = comments.length;
    }

    if (comments.length === 0) {
        commentsList.innerHTML = '<div class="text-center py-4 text-gray-500 text-sm">No comments yet. Be the first to comment!</div>';
        return;
    }

    commentsList.innerHTML = comments.map(comment => `
        <div class="flex space-x-2">
            <div class="flex-shrink-0">
                ${comment.user.avatar ?
                    `<img src="/storage/${comment.user.avatar}" alt="${comment.user.name}" class="w-6 h-6 rounded-full object-cover">` :
                    `<div class="w-6 h-6 bg-gradient-to-br from-gray-400 to-gray-600 rounded-full flex items-center justify-center">
                        <span class="text-white font-bold text-xs">${comment.user.name.charAt(0)}</span>
                    </div>`
                }
            </div>
            <div class="flex-1">
                <div class="bg-white rounded-lg px-3 py-2 shadow-sm">
                    <div class="flex items-center space-x-2 mb-1">
                        <span class="font-semibold text-sm text-gray-900">${comment.user.name}</span>
                        <span class="text-xs text-gray-500">${new Date(comment.created_at).toLocaleDateString()}</span>
                    </div>
                    <p class="text-sm text-gray-700">${comment.comment}</p>
                    ${comment.replies && comment.replies.length > 0 ? `
                    <div class="mt-2 space-y-1">
                        ${comment.replies.map(reply => `
                        <div class="flex space-x-2 ml-4 border-l-2 border-gray-200 pl-3">
                            <div class="flex-shrink-0">
                                ${reply.user.avatar ?
                                    `<img src="/storage/${reply.user.avatar}" alt="${reply.user.name}" class="w-5 h-5 rounded-full object-cover">` :
                                    `<div class="w-5 h-5 bg-gradient-to-br from-gray-400 to-gray-600 rounded-full flex items-center justify-center">
                                        <span class="text-white font-bold text-xs">${reply.user.name.charAt(0)}</span>
                                    </div>`
                                }
                            </div>
                            <div class="flex-1">
                                <div class="bg-gray-50 rounded px-2 py-1">
                                    <div class="flex items-center space-x-1 mb-1">
                                        <span class="font-medium text-xs text-gray-900">${reply.user.name}</span>
                                        <span class="text-xs text-gray-500">${new Date(reply.created_at).toLocaleDateString()}</span>
                                    </div>
                                    <p class="text-xs text-gray-700">${reply.comment}</p>
                                </div>
                            </div>
                        </div>
                        `).join('')}
                    </div>
                    ` : ''}
                    <div class="mt-1">
                        <button onclick="showReplyForm(${comment.id}, ${jobId})" class="text-xs text-yellow-600 hover:text-yellow-800">Reply</button>
                    </div>
                </div>
                <div id="reply-form-${comment.id}" class="hidden mt-2 ml-8">
                    ${document.querySelector('[data-auth]') ? `
                    <div class="flex space-x-2">
                        <input type="text" placeholder="Write a reply..." class="flex-1 px-2 py-1 text-xs border border-gray-300 rounded focus:outline-none focus:ring-1 focus:ring-yellow-500" onkeypress="handleReplyKeypress(event, ${comment.id}, ${jobId})">
                        <button onclick="postReply(${comment.id}, ${jobId})" class="px-3 py-1 bg-yellow-400 hover:bg-yellow-500 text-black text-xs rounded font-semibold transition">
                            Reply
                        </button>
                    </div>
                    ` : `
                    <div class="text-center py-1">
                        <a href="/login" class="text-yellow-600 hover:text-yellow-800 text-xs">Login to reply</a>
                    </div>
                    `}
                </div>
            </div>
        </div>
    `).join('');
}

function handleCommentKeypress(event, jobId) {
    if (event.key === 'Enter') {
        postComment(jobId);
    }
}

function postComment(jobId) {
    @auth
        const input = event.target.closest('.flex-1').querySelector('input');
        const comment = input.value.trim();

        if (!comment) return;

        fetch(`/api/v1/jobs/${jobId}/comments`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ comment: comment })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                input.value = '';
                // Reload comments
                loadComments(jobId);
                showNotification('Comment posted successfully', 'success');
            } else {
                showNotification('Failed to post comment', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Failed to post comment', 'error');
        });
    @else
        window.location.href = '{{ route("login") }}';
    @endauth
}

function showReplyForm(commentId, jobId) {
    const replyForm = document.getElementById(`reply-form-${commentId}`);
    replyForm.classList.toggle('hidden');
}

function handleReplyKeypress(event, commentId, jobId) {
    if (event.key === 'Enter') {
        postReply(commentId, jobId);
    }
}

function postReply(commentId, jobId) {
    @auth
        const input = event.target.closest('.flex').querySelector('input');
        const reply = input.value.trim();

        if (!reply) return;

        fetch(`/api/v1/jobs/${jobId}/comments`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                comment: reply,
                parent_id: commentId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                input.value = '';
                // Hide reply form
                document.getElementById(`reply-form-${commentId}`).classList.add('hidden');
                // Reload comments
                loadComments(jobId);
                showNotification('Reply posted successfully', 'success');
            } else {
                showNotification('Failed to post reply', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Failed to post reply', 'error');
        });
    @else
        window.location.href = '{{ route("login") }}';
    @endauth
}

function quickApply(jobId) {
    @auth
        if (confirm('Apply with your existing profile information?')) {
            fetch(`/api/v1/jobs/${jobId}/apply`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    cover_letter: 'Applied via quick apply from job listing'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Application submitted successfully!', 'success');
                    // Update apply button
                    const button = event.target.closest('button');
                    button.disabled = true;
                    button.innerHTML = `
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        Applied
                    `;
                    button.classList.remove('bg-gradient-to-r', 'from-black', 'to-gray-800', 'hover:from-gray-800', 'hover:to-black');
                    button.classList.add('bg-green-600', 'hover:bg-green-700');
                } else {
                    showNotification(data.message || 'Failed to apply', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Failed to apply for job', 'error');
            });
        }
    @else
        window.location.href = '{{ route("login") }}';
    @endauth
}

// Close dropdowns when clicking outside
document.addEventListener('click', function(event) {
    if (!event.target.closest('[onclick*="toggleJobOptions"]') && !event.target.closest('[id^="job-options-"]')) {
        document.querySelectorAll('[id^="job-options-"]').forEach(el => {
            el.classList.add('hidden');
        });
    }
});

function saveSearch() {
    @auth
        // Save current search parameters
        const searchData = {
            query: '{{ request("q") }}',
            location: '{{ request("location") }}',
            sector: '{{ request("sector") }}',
            experience_level: '{{ request("experience_level") }}',
            employment_type: '{{ request("employment_type") }}',
            salary_min: '{{ request("salary_min") }}',
            salary_max: '{{ request("salary_max") }}',
            remote: '{{ request("remote") }}',
            featured: '{{ request("featured") }}'
        };

        // Send to server (you would implement this endpoint)
        fetch('/api/user/saved-searches', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(searchData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Search saved successfully!');
            } else {
                alert('Failed to save search. Please try again.');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Failed to save search. Please try again.');
        });
    @else
        // Redirect to login
        window.location.href = '{{ route("login") }}';
    @endauth
}

function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg transform transition-all duration-300 translate-x-full max-w-sm`;

    // Set colors based on type
    const colors = {
        success: 'bg-green-500 text-white',
        error: 'bg-red-500 text-white',
        info: 'bg-blue-500 text-white'
    };

    notification.classList.add(...colors[type].split(' '));
    notification.innerHTML = `
        <div class="flex items-center">
            <div class="flex-1">${message}</div>
            <button onclick="this.parentElement.parentElement.remove()" class="ml-2 text-white hover:text-gray-200">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                </svg>
            </button>
        </div>
    `;

    document.body.appendChild(notification);

    // Animate in
    setTimeout(() => {
        notification.classList.remove('translate-x-full');
    }, 100);

    // Auto remove after 5 seconds
    setTimeout(() => {
        if (document.body.contains(notification)) {
            notification.classList.add('translate-x-full');
            setTimeout(() => {
                if (document.body.contains(notification)) {
                    document.body.removeChild(notification);
                }
            }, 300);
        }
    }, 5000);
}

// Auto-submit form when filters change (optional enhancement)
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('searchForm');
    const inputs = form.querySelectorAll('input, select');

    inputs.forEach(input => {
        input.addEventListener('change', function() {
            // Optional: Auto-submit on filter change
            // form.submit();
        });
    });
});
</script>