<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\JobController;
use App\Http\Controllers\Api\ApplicationController;
use App\Http\Controllers\Api\UserController;
use App\Http\Controllers\Api\SavedSearchController;
use App\Http\Controllers\Api\SubscriptionController;
use App\Http\Controllers\Api\PaymentController;
use App\Http\Controllers\Api\AdController;
use App\Http\Controllers\Api\JobSocialController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

Route::middleware('auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});

// Saved searches for the web UI (expects /api/user/saved-searches)
Route::middleware('auth:sanctum')->post('/user/saved-searches', [SavedSearchController::class, 'store']);

// Public API Routes
Route::prefix('v1')->group(function () {

    // Job Routes
    Route::get('/jobs', [JobController::class, 'index']);
    Route::get('/jobs/{job}', [JobController::class, 'show']);
    Route::get('/jobs/search/{query}', [JobController::class, 'search']);

    // Company Routes
    Route::get('/companies', [App\Http\Controllers\Api\CompanyController::class, 'index']);
    Route::get('/companies/{company}', [App\Http\Controllers\Api\CompanyController::class, 'show']);

    // Public Content
    Route::get('/sectors', function () {
        return response()->json([
            'sectors' => [
                'Technology', 'Telecom', 'Education', 'Healthcare',
                'Retail', 'Logistics', 'Hospitality', 'Agriculture',
                'Construction', 'Finance', 'Energy', 'Public Sector'
            ]
        ]);
    });

    Route::get('/locations', function () {
        return response()->json([
            'locations' => [
                'Douala', 'Yaoundé', 'Limbé', 'Bafoussam', 'Bamenda',
                'Garoua', 'Maroua', 'Ngaoundéré', 'Bertoua', 'Ebolowa'
            ]
        ]);
    });

    // Ad Routes (public for display)
    Route::get('/ads/floating', [AdController::class, 'getFloatingAds']);
    Route::get('/ads/banner/{position}', [AdController::class, 'getBannerAds']);
    Route::post('/ads/{ad}/impression', [AdController::class, 'recordImpression']);
    Route::post('/ads/{ad}/click', [AdController::class, 'recordClick']);
});

// Protected API Routes (require authentication)
Route::middleware(['auth:sanctum'])->prefix('v1')->group(function () {

    // User Profile Routes
    Route::get('/profile', [UserController::class, 'profile']);
    Route::put('/profile', [UserController::class, 'updateProfile']);
    Route::post('/profile/resume', [UserController::class, 'uploadResume']);

    // Job Seeker Routes
    Route::post('/jobs/{job}/apply', [ApplicationController::class, 'apply']);
    // Job Social Interactions
    Route::prefix('jobs/{job}')->group(function () {
        Route::post('/like', [JobSocialController::class, 'toggleLike']);
        Route::get('/comments', [JobSocialController::class, 'getComments']);
        Route::post('/comments', [JobSocialController::class, 'addComment']);
        Route::post('/share', [JobSocialController::class, 'recordShare']);
        Route::get('/stats', [JobSocialController::class, 'getStats']);
    });
    Route::post('/jobs/{job}/report', [JobController::class, 'report']);
    Route::get('/applications', [ApplicationController::class, 'myApplications']);
    Route::get('/applications/{application}', [ApplicationController::class, 'show']);

    // Company Routes
    Route::middleware(['company.owner'])->group(function () {
        Route::post('/jobs', [JobController::class, 'store']);
        Route::put('/jobs/{job}', [JobController::class, 'update']);
        Route::delete('/jobs/{job}', [JobController::class, 'destroy']);
        Route::get('/company/applications', [ApplicationController::class, 'companyApplications']);
        Route::put('/applications/{application}/status', [ApplicationController::class, 'updateStatus']);

        // Subscription Routes
        Route::get('/subscription', [SubscriptionController::class, 'show']);
        Route::get('/subscription/plans', [SubscriptionController::class, 'plans']);
        Route::post('/subscription/subscribe', [SubscriptionController::class, 'subscribe']);
        Route::post('/subscription/cancel', [SubscriptionController::class, 'cancel']);
        Route::post('/subscription/renew', [SubscriptionController::class, 'renew']);
        Route::get('/subscription/transactions', [SubscriptionController::class, 'transactions']);
    });

    // Payment Routes
    Route::middleware(['auth:sanctum'])->group(function () {
        Route::post('/payments/initiate', [PaymentController::class, 'initiate']);
        Route::get('/payments/{transactionId}/status', [PaymentController::class, 'status']);
        Route::get('/payments/history', [PaymentController::class, 'history']);
    });

    // Admin Routes
    Route::middleware(['admin'])->prefix('admin')->group(function () {
        Route::get('/stats', [App\Http\Controllers\Api\AdminController::class, 'stats']);
        Route::get('/users', [App\Http\Controllers\Api\AdminController::class, 'users']);
        Route::get('/jobs', [App\Http\Controllers\Api\AdminController::class, 'jobs']);
        Route::get('/applications', [App\Http\Controllers\Api\AdminController::class, 'applications']);
        Route::delete('/users/{user}', [App\Http\Controllers\Api\AdminController::class, 'deleteUser']);
        Route::delete('/jobs/{job}', [App\Http\Controllers\Api\AdminController::class, 'deleteJob']);
        Route::post('/payments/{transactionId}/refund', [PaymentController::class, 'refund']);
    });
});
