<?php

namespace Tests\Unit\Services;

use Tests\TestCase;
use App\Models\Ad;
use App\Models\AdPlan;
use App\Services\AdMonitoringService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AdMonitoringServiceTest extends TestCase
{
    use RefreshDatabase;

    private AdMonitoringService $monitoringService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->monitoringService = app(AdMonitoringService::class);
    }

    /** @test */
    public function it_calculates_system_metrics_correctly()
    {
        // Create some test data
        $plan = AdPlan::factory()->create(['cost_per_click' => 100]);
        
        $ad = Ad::factory()->create([
            'status' => 'active',
            'plan_id' => $plan->id
        ]);

        // Add some clicks and impressions
        DB::table('ad_impressions')->insert([
            'ad_id' => $ad->id,
            'created_at' => now(),
            'user_id' => 1,
            'ip_hash' => 'test'
        ]);

        DB::table('ad_clicks')->insert([
            'ad_id' => $ad->id,
            'created_at' => now(),
            'user_id' => 1,
            'ip_hash' => 'test'
        ]);

        $metrics = $this->monitoringService->getSystemMetrics();

        $this->assertEquals(1, $metrics['total_active_ads']);
        $this->assertEquals(1, $metrics['total_impressions_24h']);
        $this->assertEquals(1, $metrics['total_clicks_24h']);
        $this->assertEquals(100, $metrics['system_ctr']);
        $this->assertEquals(100, $metrics['total_revenue_24h']);
    }

    /** @test */
    public function it_detects_performance_alerts()
    {
        // Create an ad with low CTR
        Ad::factory()->create([
            'status' => 'active',
            'impressions' => 1000,
            'clicks' => 5 // 0.5% CTR
        ]);

        // Create an ad with exhausted budget
        Ad::factory()->create([
            'status' => 'active',
            'budget' => 1000,
            'spent' => 1000
        ]);

        // Create an ad about to expire
        Ad::factory()->create([
            'status' => 'active',
            'end_date' => now()->addDays(2)
        ]);

        $alerts = $this->monitoringService->getPerformanceAlerts();

        $this->assertCount(3, $alerts);
        $this->assertStringContainsString('CTR below 1%', $alerts[0]['message']);
        $this->assertStringContainsString('exhausted their budget', $alerts[1]['message']);
        $this->assertStringContainsString('expiring in the next 3 days', $alerts[2]['message']);
    }

    /** @test */
    public function it_monitors_abnormal_ad_activity()
    {
        $plan = AdPlan::factory()->create(['cost_per_click' => 100]);
        
        $ad = Ad::factory()->create([
            'status' => 'active',
            'plan_id' => $plan->id,
            'budget' => 3000, // Daily budget should be 100
            'spent' => 0
        ]);

        // Simulate high CTR activity
        for ($i = 0; $i < 100; $i++) {
            DB::table('ad_impressions')->insert([
                'ad_id' => $ad->id,
                'created_at' => now(),
                'user_id' => $i,
                'ip_hash' => "test{$i}"
            ]);
        }

        for ($i = 0; $i < 30; $i++) {
            DB::table('ad_clicks')->insert([
                'ad_id' => $ad->id,
                'created_at' => now(),
                'user_id' => $i,
                'ip_hash' => "test{$i}"
            ]);
        }

        $this->monitoringService->monitorAdActivity($ad);

        // Check if warning was logged (you may need to customize this based on your logging setup)
        $this->assertDatabaseHas('system_logs', [
            'level' => 'warning',
            'message' => 'Unusually high CTR detected'
        ]);
    }

    /** @test */
    public function it_reports_system_health_status()
    {
        // Create a healthy system state
        Ad::factory()->count(15)->create(['status' => 'active']);

        $health = $this->monitoringService->getSystemHealth();

        $this->assertEquals('healthy', $health['status']);
        $this->assertEmpty($health['issues']);
        $this->assertArrayHasKey('metrics', $health);

        // Create an unhealthy state
        Ad::factory()->count(14)->update(['status' => 'inactive']);

        $health = $this->monitoringService->getSystemHealth();

        $this->assertEquals('warning', $health['status']);
        $this->assertNotEmpty($health['issues']);
        $this->assertStringContainsString('Low number of active ads', $health['issues'][0]);
    }
}