<?php

namespace Tests\Unit\Services;

use Tests\TestCase;
use App\Models\Ad;
use App\Models\User;
use App\Services\AdService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;

class AdServiceTest extends TestCase
{
    use RefreshDatabase;

    private AdService $adService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->adService = app(AdService::class);
    }

    /** @test */
    public function it_returns_active_ads_for_given_position()
    {
        // Create test ads
        $activeAd = Ad::factory()->create([
            'status' => 'active',
            'position' => 'sidebar',
            'type' => 'banner',
            'start_date' => now()->subDay(),
        ]);

        $inactiveAd = Ad::factory()->create([
            'status' => 'inactive',
            'position' => 'sidebar',
            'type' => 'banner',
        ]);

        $ads = $this->adService->getActiveAds(null, 'banner', 'sidebar');

        $this->assertCount(1, $ads);
        $this->assertTrue($ads->contains($activeAd));
        $this->assertFalse($ads->contains($inactiveAd));
    }

    /** @test */
    public function it_prevents_duplicate_impressions_from_same_ip()
    {
        $ad = Ad::factory()->create(['status' => 'active']);

        // First impression should succeed
        $result1 = $this->adService->recordImpression($ad);
        $this->assertTrue($result1);

        // Second impression from same IP should be blocked
        $result2 = $this->adService->recordImpression($ad);
        $this->assertFalse($result2);
    }

    /** @test */
    public function it_prevents_duplicate_clicks_from_same_ip()
    {
        $ad = Ad::factory()->create(['status' => 'active']);

        // First click should succeed
        $result1 = $this->adService->recordClick($ad);
        $this->assertTrue($result1);

        // Second click from same IP should be blocked
        $result2 = $this->adService->recordClick($ad);
        $this->assertFalse($result2);
    }

    /** @test */
    public function it_calculates_ad_statistics_correctly()
    {
        $ad = Ad::factory()->create([
            'status' => 'active',
            'impressions' => 1000,
            'clicks' => 50,
            'budget' => 10000,
            'spent' => 5000,
        ]);

        $stats = $this->adService->getAdStats($ad);

        $this->assertEquals(1000, $stats['total_impressions']);
        $this->assertEquals(50, $stats['total_clicks']);
        $this->assertEquals(5, $stats['click_through_rate']);
        $this->assertEquals(5000, $stats['remaining_budget']);
    }

    /** @test */
    public function it_handles_ad_targeting_rules()
    {
        $user = User::factory()->create([
            'role' => 'company_owner',
            'location' => 'Douala'
        ]);

        $ad = Ad::factory()->create([
            'status' => 'active',
            'targeting' => [
                'roles' => ['company_owner'],
                'locations' => ['Douala', 'Yaoundé']
            ]
        ]);

        $ads = $this->adService->getActiveAds($user);
        $this->assertTrue($ads->contains($ad));

        // Test with non-matching user
        $otherUser = User::factory()->create([
            'role' => 'job_seeker',
            'location' => 'Bamenda'
        ]);

        $ads = $this->adService->getActiveAds($otherUser);
        $this->assertFalse($ads->contains($ad));
    }

    /** @test */
    public function it_activates_scheduled_ads()
    {
        // Create a scheduled ad that should be activated
        Ad::factory()->create([
            'status' => 'scheduled',
            'start_date' => now()->subHour()
        ]);

        // Create a scheduled ad that should not be activated yet
        Ad::factory()->create([
            'status' => 'scheduled',
            'start_date' => now()->addHour()
        ]);

        $activatedCount = $this->adService->activateScheduledAds();

        $this->assertEquals(1, $activatedCount);
        $this->assertEquals(1, Ad::where('status', 'active')->count());
    }

    /** @test */
    public function it_deactivates_expired_ads()
    {
        // Create an expired active ad
        Ad::factory()->create([
            'status' => 'active',
            'end_date' => now()->subHour()
        ]);

        // Create an active ad that should not expire yet
        Ad::factory()->create([
            'status' => 'active',
            'end_date' => now()->addHour()
        ]);

        $deactivatedCount = $this->adService->deactivateExpiredAds();

        $this->assertEquals(1, $deactivatedCount);
        $this->assertEquals(1, Ad::where('status', 'inactive')->count());
    }

    /** @test */
    public function it_pauses_ads_with_exhausted_budget()
    {
        // Create an ad that has exhausted its budget
        Ad::factory()->create([
            'status' => 'active',
            'budget' => 1000,
            'spent' => 1000
        ]);

        // Create an ad that still has budget
        Ad::factory()->create([
            'status' => 'active',
            'budget' => 1000,
            'spent' => 500
        ]);

        $pausedCount = $this->adService->pauseExhaustedAds();

        $this->assertEquals(1, $pausedCount);
        $this->assertEquals(1, Ad::where('status', 'inactive')->count());
    }

    /** @test */
    public function it_validates_targeting_rules()
    {
        $targeting = [
            'roles' => ['admin', 'invalid_role'],
            'company_sizes' => ['1-10', 'invalid_size'],
            'locations' => ['Douala', 'invalid_location']
        ];

        $errors = $this->adService->validateTargeting($targeting);

        $this->assertCount(3, $errors);
        $this->assertStringContainsString('Invalid roles', $errors[0]);
        $this->assertStringContainsString('Invalid company sizes', $errors[1]);
        $this->assertStringContainsString('Invalid locations', $errors[2]);
    }
}